/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef SFM_H
#define SFM_H

#include <vector>
#include <memory>
#include "../tools/LinAlg.h"
#include "BundleAdjustment/BundleAdjustment.h"
#include "../cudaInterface/CudaSift.h"
//#include "FeaturePointAlignmentEstimator.h"
#include "../tools/CPUStopWatch.h"

#include "../config/SfMConfig.h"

namespace SFM {


class InternalCameraCalibration;
class Track;
class TrackObservation;
class Frame;


//#define SFM_PERFORM_PATCH_ALIGNMENT

class InternalCameraCalibration;

class DescriptorMatchingStats {
    public:
        enum MatchMode {
            MATCH_INITIAL           = 0,
            MATCH_UNCONSTRAINED     = 1,
            MATCH_EPIPOLAR          = 2,
            MATCH_LOCATION          = 3,
            MATCH_MODE_COUNT        = 4
        };

        DescriptorMatchingStats();


        void startMatching();
        void finishedMatching(MatchMode mode, uint64_t numMatchingOps);

        inline uint64_t getNumMatchingOperations(MatchMode mode) const { return m_numMatchingOperations[mode]; }
        inline uint64_t getNanosecondsSpendMatching(MatchMode mode) const { return m_nanosecondsSpendMatching[mode]; }
    private:
        uint64_t m_numMatchingOperations[MATCH_MODE_COUNT];
        uint64_t m_nanosecondsSpendMatching[MATCH_MODE_COUNT];
        Engine::CPUStopWatch m_stopWatch;
        bool m_currentlyMatching;
};

/**
 * @brief Main class responsible for the entire structure from motion process.
 *
 * @ingroup SFMBackend_Group
 */
class SFM
{
    public:
        class PerformanceStatistics {
            public:
                enum Timer {
                    TIMER_BUNDLE_ADJUSTMENT   = 0,
                    TIMER_ADJUST_SEMI_ACTIVE   = 1,
                    TIMER_TRACK_ALIGNMENT     = 2,
                    TIMER_NEW_TRACK_GATHERING = 3,
                    TIMER_NEW_TRACK_TESTING   = 4,
                    TIMER_NEW_OBS_GATHERING   = 5,
                    TIMER_NEW_OBS_TESTING     = 6,
                    TIMER_NEW_FRAME_PHASE1    = 7,
                    TIMER_NEW_FRAME_PHASE2_GATHER    = 8,
                    TIMER_NEW_FRAME_PHASE2_TEST    = 9,
                    TIMER_NEW_FRAME_PHASE2_RANSAC    = 10,
                    TIMER_FEATURE_EXTRACTION  = 11,
                    TIMER_REMOVE_DUPLICATES   = 12,
                    NUM_TIMERS                = 13
                };
                static const char *TimerToStr(Timer timer);

                struct TimeFrame {
                    uint64_t m_interval[2];

                    uint64_t m_timerSum[NUM_TIMERS];

                    unsigned numActiveTracks;
                    unsigned numActiveObs;
                };

                PerformanceStatistics();

                void startNewFrame();

                void startTimer(Timer timer);
                void stopTimer(Timer timer);

                void dump();
            private:
                void endFrame();
                Engine::CPUStopWatch m_overallStopWatch;
                Engine::CPUStopWatch m_subTimer[NUM_TIMERS];

                std::vector<TimeFrame> m_timeFrames;
                TimeFrame m_currentTimeFrame;
        };

        SFM(const config::SfMConfig &config = config::SfMConfig());
        ~SFM();
        void clear();

        unsigned testbenchAddCalibration(const LinAlg::Matrix4x4f &priorProjectionMatrix,
                                         const BundleAdjustment::RadialDistortionParametrization &radialDistortion,
                                         float aspectRatio);
        unsigned testbenchAddFrame(unsigned calibIndex, std::vector<CudaSift::FeaturePoint> &featurePoints, unsigned width, unsigned height, const LinAlg::Matrix4x4f &viewMatrix);
        unsigned testbenchAddTrack(const LinAlg::Vector4f &wsPos);
        unsigned testbenchAddTrackObservation(unsigned trackIndex, unsigned cameraIndex, unsigned featurePointIndex);
        void testbenchMakeAllTracksActive();
        void testbenchMakeAllFramesActive();


        struct ExtractFeaturePointsProgressReport {
            std::string imageFilename;
            unsigned imageNumber;
            unsigned totalImageNumber;
            uint64_t numfeaturePointsSoFar;
        };
        void extractFeaturePoints(const std::function<void(const ExtractFeaturePointsProgressReport&)> &progressCallback = std::function<void(const ExtractFeaturePointsProgressReport&)>());

        void initSingleCameraPath(const LinAlg::Matrix4x4f &priorProjectionMatrix,
                                  const BundleAdjustment::RadialDistortionParametrization &radialDistortion,
                                  float aspectRatio,
                                  const std::vector<std::string> &filenames);
        void buildInitialPose(unsigned frame1, unsigned frame2);

        inline const ChunkedArray<Track> &getTrackList() const { return m_trackList; }
        inline const std::vector<std::unique_ptr<Frame> > &getFrames() const { return m_frames; }
        inline const std::vector<std::unique_ptr<InternalCameraCalibration> > &getInternalCameraCalibrations() const { return m_internalCameraCalibrations; }

        void iterateBA(unsigned iterations, bool runSemiActive = true);

        void operate();

void dumpTrackObsForAnalysis(const std::string &filename);

        inline void dumpPerfStats() { m_performanceStatistics.dump(); }

        float computeSumSqrProjectionErrors();
        float computeAvgSqrProjectionErrors();

        inline uint32_t getTimestampClock() const { return m_timestampClock; }

        enum OpState {
            OP_STATE_INITIATE,
            OP_STATE_ADDPATCHES,
            OP_STATE_REMOVEOUTLIERS,
            OP_STATE_ADDVIEW,
            OP_STATE_DONE
        };
        inline OpState getState() const { return m_state; }


        void selectBASubset();
        void recomputeTrackWeights();
    protected:
        config::SfMConfig m_config;

        void activateNextFrames();
        void findAdditionalObservations();
        void removeOutliers();

        void readbackBAData();

        #ifdef SFM_PERFORM_PATCH_ALIGNMENT
        void computeMissingFeaturePointAlignments(bool measureTime = true);
        void updateTrackNormals();
        #endif

        void mergeDuplicateTracks();


        void readjustSemiActiveTracks();


        void readjustSemiActiveTrackRange(unsigned start, unsigned count);
        void checkMajorTrackChanges();

        unsigned m_removeOutlierRun;

        OpState m_state;

        std::vector<std::unique_ptr<InternalCameraCalibration> > m_internalCameraCalibrations;
        std::vector<std::unique_ptr<Frame> > m_frames;
        ChunkedArray<Track> m_trackList;
        #ifdef SFM_PERFORM_PATCH_ALIGNMENT
        FeaturePointAlignmentEstimator m_featurePointAlignmentEstimator;
        #endif

        BundleAdjustment m_bundleAdjustment;

        PerformanceStatistics m_performanceStatistics;
        DescriptorMatchingStats m_descriptorMatchingStats;

        unsigned m_allViewsReconstructedExtraRounds;

        uint32_t m_timestampClock;

        unsigned m_numCyclesWithoutNewImages;
};

}

#endif // SFM_H
