/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef TRACK_H
#define TRACK_H

#include "../tools/LinAlg.h"
#include <memory>
#include "TrackMatchingDescriptor.h"
#include "TrackObservation.h"
#include "../tools/ChunkedArray.hpp"
#include "BundleAdjustment/BundleAdjustment.h"

namespace SFM {

class BATrackList;

/**
 * @brief Represents a track for the structure from motion process.
 *
 * @ingroup SFMBackend_Group
 */
class Track
{
    public:
        /// The state that a track can be in
        enum State {
            STATE_ACTIVE,           ///< The track is valid and actively participates in Bundle Adjustment (it is in the subset).
            STATE_SEMI_ACTIVE,      ///< The track is valid but does not actively participate in Bundle Adjustment (but gets optimized by the structure only BA).
            STATE_DISABLED          ///< The track has become invalid after its observations were marked as faulty.
        };

        Track();
        ~Track();
        void setup(BundleAdjustment *ba);
        void switchState(State newState);

        unsigned addObservation(Frame *frame, unsigned framePatchIndex);
        void setReferenceObservation(TrackObservation *obs);
        void chooseNewReferenceObservation();
        inline TrackObservation *getReferenceObservation() { return m_referenceObservation; }
        inline const TrackObservation *getReferenceObservation() const { return m_referenceObservation; }

        inline void setWSPosition(const LinAlg::Vector4f &WSPosition) { m_lastWSPositionEstimate = WSPosition; }
        const LinAlg::Vector4f &getLastWSPositionEstimate() const { return m_lastWSPositionEstimate; }

        inline float getSize() const { return m_size; }
        inline const LinAlg::Vector3f &getNormal() const { return m_normal; }
        inline const LinAlg::Matrix3x3f &getOrientation() const { return m_orientation; }
        inline void setNeedsNormalUpdate() { m_needsNormalUpdate = true; }
        inline bool needsNormalUpdate() const { return m_needsNormalUpdate; }

        void updateNormal();


        inline BundleAdjustment::TrackHandle getBAHandle() const { return m_baTrackHandle; }
        void readBackFromBA();

        inline const ChunkedArray<TrackObservation, 16> &getObservations() const { return m_observations; }
        inline ChunkedArray<TrackObservation, 16> &getObservations() { return m_observations; }

        void obsChanged();

        void updateTrackMatchingDescriptor(const uint32_t *data);
        TrackMatchingDescriptor *getTrackMatchingDescriptor() { return m_trackMatchingDescriptor.get(); }

        inline float getTrackWeight() const { return m_trackWeight; }
        void setTrackWeight(float trackWeight);

        inline State getState() const { return m_state; }

        inline void addBASubsetVote() { m_baSubesetVotes++; }
        inline void clearBASubsetVotes() { m_baSubesetVotes = 0; }
        inline unsigned getBASubsetVotes() { return m_baSubesetVotes; }

        inline BundleAdjustment *getBA() { return m_ba; }

        inline uint32_t getLastMajorChangeTimestamp() const { return m_majorChangeTimestamp.m_lastMajorChangeTimestamp; }
        void checkForMajorChange(uint32_t currentTimestamp);
    protected:
        BundleAdjustment *m_ba;
        State m_state;
        unsigned m_baSubesetVotes;
        bool m_needsNormalUpdate;
        LinAlg::Vector4f m_lastWSPositionEstimate;
        LinAlg::Vector3f m_normal;
        LinAlg::Matrix3x3f m_orientation;
        float m_size;
        float m_trackWeight;
        BundleAdjustment::TrackHandle m_baTrackHandle;
        std::unique_ptr<TrackMatchingDescriptor> m_trackMatchingDescriptor;
        ChunkedArray<TrackObservation, 16> m_observations;
        TrackObservation *m_referenceObservation;

        void activateBA();
        void deactivateBA();

        struct MajorChangeTimestamp {
            uint32_t m_lastMajorChangeTimestamp;
            bool m_observationsChangedSinceLastTimestampUpdate;
            LinAlg::Vector4f m_WSPositionEstimate;

            MajorChangeTimestamp() : m_lastMajorChangeTimestamp(0), m_observationsChangedSinceLastTimestampUpdate(true) { }

            inline void observationsChanged() { m_observationsChangedSinceLastTimestampUpdate = true; }

            void checkUpdateTimestamp(uint32_t currentTimestamp, const LinAlg::Vector4f &lastWSPositionEstimate, float size);
        };
        MajorChangeTimestamp m_majorChangeTimestamp;

        TrackObservation *chooseBestReferenceObservation(float *score = NULL);

};

}

#endif // TRACK_H
