/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CUDATEXTUREMEMORY_H
#define CUDATEXTUREMEMORY_H



#include <cuda.h>
#include <vector>
#include "../tools/LinAlg.h"

namespace CudaUtils {


class BaseCudaTextureMemory
{
    public:
        BaseCudaTextureMemory();
        ~BaseCudaTextureMemory();

        static unsigned getFormatSize(CUarray_format format);

        inline unsigned getWidth() const { return m_descriptor.Width; }
        inline unsigned getHeight() const { return m_descriptor.Height; }
        inline unsigned getDepth() const { return m_descriptor.Depth; }

        void syncUploadAll(const void *src, unsigned pitchInBytes, unsigned heightInPixels = 0);
        void syncDownloadAll(void *dst, unsigned pitchInBytes, unsigned heightInPixels = 0);

        void syncDownloadSingleLayer(void *dst, unsigned pitchInBytes, unsigned layer, unsigned heightInPixels = 0);

        inline CUarray getHandle() { return m_handle; }

    protected:
        CUDA_ARRAY3D_DESCRIPTOR m_descriptor;
        CUarray m_handle;
};


class CudaTextureMemory : public BaseCudaTextureMemory
{
    public:
        CudaTextureMemory();
        ~CudaTextureMemory();

        void resize(unsigned width, unsigned height, unsigned depth, CUarray_format format, unsigned numComponents, unsigned flags);
    protected:
        bool m_handleValid;
};

class CudaMipmappedTexture;

class CudaMipmappedTextureLevelMemory : public BaseCudaTextureMemory
{
    public:
    protected:
        CudaMipmappedTexture *m_mipmappedTexture;
        unsigned m_level;

        void setup(CudaMipmappedTexture *mipmappedTexture, unsigned level);

        friend class CudaMipmappedTexture;
};


class CudaMipmappedTexture
{
    public:
        CudaMipmappedTexture();
        ~CudaMipmappedTexture();

        void resize(unsigned width, unsigned height, unsigned depth, CUarray_format format, unsigned numComponents, unsigned flags, unsigned numLevel = -1);

        inline unsigned getNumLevel() const { return m_level.size(); }
        inline CudaMipmappedTextureLevelMemory &getLevel(unsigned level) { return m_level[level]; }

        inline CUmipmappedArray getHandle() { return m_handle; }

        static unsigned getMaxMipLevelCount(unsigned width, unsigned height);
    protected:
        bool m_handleValid;
        CUDA_ARRAY3D_DESCRIPTOR m_descriptor;
        CUmipmappedArray m_handle;

        std::vector<CudaMipmappedTextureLevelMemory> m_level;
};

}

#endif // CUDATEXTUREMEMORY_H
