/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef CHUNKEDSOA_H
#define CHUNKEDSOA_H

#include <vector>
#include <memory>

/** @addtogroup Codebase_Group
 *  @{
 */

class ChunkedSOA
{
    public:
        ChunkedSOA(unsigned numMembers, const unsigned *memberSizes);
        ~ChunkedSOA();

        enum {
            CHUNK_SIZE_LOG2 = 8,
            CHUNK_SIZE = 1 << CHUNK_SIZE_LOG2,
            INTRA_CHUNK_INDEX_MASK = CHUNK_SIZE-1,
        };

        void clear();
        void resize(unsigned numElements);
        void reserve(unsigned numElements);
        inline unsigned size() const { return m_size; }

        inline void *getMember(unsigned index, unsigned member) {
            Chunk *chunk = m_chunks[index >> CHUNK_SIZE_LOG2].get();
            unsigned intraIndex = index & INTRA_CHUNK_INDEX_MASK;
            return chunk->ptr + m_members[member].offset + intraIndex * m_members[member].stride;
        }

        inline const void *getMember(unsigned index, unsigned member) const {
            Chunk *chunk = m_chunks[index >> CHUNK_SIZE_LOG2].get();
            unsigned intraIndex = index & INTRA_CHUNK_INDEX_MASK;
            return chunk->ptr + m_members[member].offset + intraIndex * m_members[member].stride;
        }

        template<typename type>
        inline type &get(unsigned index, unsigned member) {
            return *((type*)getMember(index, member));
        }

        template<typename type>
        inline const type &get(unsigned index, unsigned member) const {
            return *((const type*)getMember(index, member));
        }

    protected:
        unsigned m_size;

        unsigned m_chunkSize;
        struct Member {
            unsigned offset;
            unsigned stride;
            unsigned size;
        };
        std::vector<Member> m_members;

        struct Chunk {
            unsigned char *ptr;
            Chunk(unsigned size);
            ~Chunk();
        };
        std::vector<std::unique_ptr<Chunk> > m_chunks;
};

/// @}

#endif // CHUNKEDSOA_H
