/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef _HOUSEHOLDERREDUCTION_HPP_
#define _HOUSEHOLDERREDUCTION_HPP_

/**
 * @file
 * @author Andreas Ley
 */


#include "LinAlg.h"
#include "TriDiagonalSymmetric.hpp"


namespace LinAlg {


/** @addtogroup Codebase_Group
 *  @{
 */


// Adapted from Numerical Recipies

template<unsigned Dim, typename Type, bool ProduceTransformation>
void HouseholderReduction(const Matrix<Dim, Dim, Type> &Source, TriDiagonalSymmetricMatrix<Dim, Type> &TriDiag, Matrix<Dim, Dim, Type> * __restrict Q = NULL)
{
    Matrix<Dim, Dim, Type> &z = ProduceTransformation?(*Q):(*new Matrix<Dim, Dim, Type>(Source));
    if (ProduceTransformation)
        z = Source;

    Type d[Dim];
    Type e[Dim];

    for (int i = Dim-1; i > 0; i--) {
        int l = i-1;
        Type scale = Type(0);
        Type h = Type(0);
        if (l > 0) {
            for (unsigned k = 0; k < (unsigned)i; k++)
                scale += std::abs(z[i][k]);

            if (scale == Type(0)) // skip transformation
                e[i] = z[i][l];
            else {
                for (unsigned k = 0; k < (unsigned)i; k++) {
                    z[i][k] /= scale;
                    h += z[i][k] * z[i][k];
                }
                {
                    Type f = z[i][l];
                    Type g = (f >= Type(0))?-std::sqrt(h):std::sqrt(h);
                    e[i] = scale * g;
                    h -= f * g;
                    z[i][l] = f - g;
                }
                {
                    Type f = Type(0);
                    for (unsigned j = 0; j < (unsigned)i; j++) {
                        if (ProduceTransformation)
                            z[j][i] = z[i][j] / h;
                        Type g = Type(0);
                        for (unsigned k = 0; k < j+1; k++)
                            g += z[j][k]*z[i][k];
                        for (unsigned k = j+1; k < (unsigned)i; k++)
                            g += z[k][j]*z[i][k];
                        e[j] = g / h;
                        f += e[j] * z[i][j];

                    }
                    Type hh = f / (h+h);
                    for (unsigned j = 0; j < (unsigned)i; j++) {
                        Type f = z[i][j];
                        Type g = e[j] - hh * f;
                        e[j] = g;
                        for (unsigned k = 0; k < j+1; k++) {
                            z[j][k] -= f * e[k] + g * z[i][k];
                        }
                    }
                }
            }
        } else {
            e[i] = z[i][l];
        }
        d[i] = h;
    }
    if (ProduceTransformation)
        d[0] = Type(0);

    //e[0] = Type(0); // neccessary?
    for (unsigned i = 0; i < Dim; i++) {
        if (ProduceTransformation) {
            if (d[i] != Type(0)) {
                for (unsigned j = 0; j < (unsigned)i; j++) {
                    Type g = Type(0);
                    for (unsigned k = 0; k < (unsigned)i; k++)
                        g += z[i][k]*z[k][j];
                    for (unsigned k = 0; k < (unsigned)i; k++)
                        z[k][j] -= g * z[k][i];
                }
            }
            TriDiag.Diagonal()[i] = z[i][i];
            z[i][i] = Type(1);
            for (unsigned j = 0; j < i; j++)
                z[j][i] = z[i][j] = Type(0);

        } else {
            TriDiag.Diagonal()[i] = z[i][i];
        }
    }
    for (unsigned i = 1; i < Dim; i++)
        TriDiag.OffDiagonal()[i-1] = e[i];

    if (!ProduceTransformation)
        delete &z;
}


template<unsigned Dim, typename Type>
void computeEigenValues(const Matrix<Dim, Dim, Type> &Source, Vector<Dim, Type> &eigenValues, Matrix<Dim, Dim, Type> *eigenVectors = NULL)
{
    if (eigenVectors != NULL) {
        TriDiagonalSymmetricMatrix<Dim, Type> triDiag;
        HouseholderReduction<Dim, Type, true>(Source, triDiag, eigenVectors);

        try {
            triDiag.computeEigenValues<true>(eigenValues, eigenVectors);
        } catch (...) {
/*
            Matrix<Dim, Dim, Type> tri = triDiag.inflate();
            std::cout << "Error for matrix: " << std::endl;
            for (unsigned i = 0; i < Dim; i++) {
                for (unsigned j = 0; j < Dim; j++) {
                    std::cout << " " << Source[i][j];
                }
                if (i < Dim-1)
                    std::cout << ";" << std::endl;
            }
            std::cout << std::endl;
            std::cout << "tridiag matrix: " << std::endl;
            for (unsigned i = 0; i < Dim; i++) {
                for (unsigned j = 0; j < Dim; j++) {
                    std::cout << " " << tri[i][j];
                }
                if (i < Dim-1)
                    std::cout << ";" << std::endl;
            }
            std::cout << std::endl;
            try {
                triDiag.computeEigenValues<false>(eigenValues, NULL);
            } catch (...) {
                std::cout << "buuuh" << std::endl;
            }
            */

            throw;
        }
    } else {
        TriDiagonalSymmetricMatrix<Dim, Type> triDiag;
        HouseholderReduction<Dim, Type, false>(Source, triDiag);
        triDiag.computeEigenValues<false>(eigenValues);
    }
}

/// @}

}

#endif // _HOUSEHOLDERREDUCTION_HPP_
