/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * @file LinAlg.h
 * @brief Shiny new OO-templated vecmath library.
 * @author Andy
 */

#ifndef _LINALG_H_
#define _LINALG_H_

#include <string>
#include <sstream>
#include <stdarg.h>
#include <stdexcept>
#include <iostream>

#include <math.h>
#include <cmath>
#include <vector>


/**
 * @ingroup Codebase_Group
 * @brief Lots of scary linear algebra math in here...
 * @details ...or at least it will be in here some day.
 */
namespace LinAlg {

/** @addtogroup Codebase_Group
 *  @{
 */


/**
 * @brief Templated implementation of a vector class.
 * @param dimension The vector's dimension (= number of elements).
 * @param type The vector's basetype (= type of elements).
 */
template <unsigned dimension, typename type, bool transposed = false>
class Vector
{
    public:
        /// direct access to the elements, in case that pointers are needed. Use the [] operator otherwise.
        type element[dimension];

        /// Fills all elements with "0".
        Vector() {
            for (unsigned i = 0; i < dimension; i++)
                element[i] = type(0);
        };

        enum NoInitializationEnum {
            NO_INITIALIZATION
        };

        Vector(NoInitializationEnum) {
        };

        /*
        /// Variadic constructor setting each element to a supplied parameter. You MUST supply one parameter for each element.
        Vector(type first, ...) {
            va_list ap;
            va_start(ap, first);
            element[0] = first;
            for (int i = 1; i < dimension; i++)
                element[i] = va_arg(ap, type);
            va_end(ap);
        };
        /// Variadic function setting each element to a supplied parameter. You MUST supply one parameter for each element.
        void Fill(type first, ...) {
            va_list ap;
            va_start(ap, first);
            element[0] = first;
            for (int i = 1; i < dimension; i++)
                element[i] = va_arg(ap, type);
            va_end(ap);
        };
*/
        /// Access to the elements in an array-like fashion
        type& operator[](unsigned i) {
            return element[i];
        };
        /// Const access to the elements in an array-like fashion
        const type& operator[](unsigned i) const {
            return element[i];
        };
        /// Add other to this vector, component wise.
        Vector<dimension, type, transposed>& operator+=(const Vector<dimension, type, transposed> &other) {
            for (unsigned i = 0; i < dimension; i++)
                element[i] += other.element[i];

            return *this;
        };
        /// Subtract other from this vector, component wise.
        Vector<dimension, type, transposed>& operator-=(const Vector<dimension, type, transposed> &other) {
            for (unsigned i = 0; i < dimension; i++)
                element[i] -= other.element[i];

            return *this;
        };
        /// Multiply this vector by f, component wise.
        Vector<dimension, type, transposed>& operator*=(const type f) {
            for (unsigned i = 0; i < dimension; i++)
                element[i] *= f;

            return *this;
        };
        /// Divide this vector by f, component wise.
        Vector<dimension, type, transposed>& operator/=(const type f) {
            for (unsigned i = 0; i < dimension; i++)
                element[i] /= f;

            return *this;
        };
        /// Copy from Vector.
        Vector<dimension, type, transposed>& operator=(const Vector<dimension, type, transposed> &other) {
            for (unsigned i = 0; i < dimension; i++)
                element[i] = other.element[i];

            return *this;
        };
        /// Copy from array.
        Vector<dimension, type, transposed>& operator=(const type *arr) {
            for (unsigned i = 0; i < dimension; i++)
                element[i] = arr[i];

            return *this;
        };
        /// Equal, if all components are equal. Not very usefull for floats...
        bool operator==(const Vector<dimension, type, transposed> &other) const {
            for (unsigned i = 0; i < dimension; i++)
                if (element[i] != other.element[i])
                    return false;

            return true;
        };
        /// Defines a total sorting order for std::map etc.
        bool operator<(const Vector<dimension, type, transposed> &other) const {
            for (unsigned i = 0; i < dimension; i++) {
                if (element[i] < other.element[i])
                    return true;
                if (element[i] > other.element[i])
                    return false;
            }
            return false;
        };
        /// Not equal.
        bool operator!=(const Vector<dimension, type, transposed> &other) const {
            return !operator==(other);
        };
        /// Returns component-wise sum.
        const Vector<dimension, type, transposed> operator+(const Vector<dimension, type, transposed> &other) const {
            Vector<dimension, type, transposed> vec(NO_INITIALIZATION);
            for (unsigned i = 0; i < dimension; i++)
                vec[i] = element[i] + other.element[i];

            return vec;
        };

        const Vector<dimension, type, transposed> mad(const Vector<dimension, type, transposed> &mul, const Vector<dimension, type, transposed> &add) const {
            Vector<dimension, type, transposed> vec(NO_INITIALIZATION);
            for (unsigned i = 0; i < dimension; i++)
                vec[i] = element[i] * mul[i] + add[i];

            return vec;
        };
        /// Returns component-wise difference.
        const Vector<dimension, type, transposed> operator-(const Vector<dimension, type, transposed> &other) const {
            Vector<dimension, type, transposed> vec(NO_INITIALIZATION);
            for (unsigned i = 0; i < dimension; i++)
                vec[i] = element[i] - other.element[i];

            return vec;
        };
        /// Returns component-wise multiplied.
        const Vector<dimension, type, transposed> operator*(const type f) const {
            Vector<dimension, type, transposed> vec(NO_INITIALIZATION);
            for (unsigned i = 0; i < dimension; i++)
                vec[i] = element[i] * f;

            return vec;
        };
        /// Returns component-wise divided.
        const Vector<dimension, type, transposed> operator/(const type f) const {
            Vector<dimension, type, transposed> vec(NO_INITIALIZATION);
            for (unsigned i = 0; i < dimension; i++)
                vec[i] = element[i] / f;

            return vec;
        };
        /// Returns negated.
        const Vector<dimension, type, transposed> negated() const {
            Vector<dimension, type, transposed> vec(NO_INITIALIZATION);
            for (unsigned i = 0; i < dimension; i++)
                vec[i] = -element[i];
            return vec;
        };
        /// Returns reciprocal.
        const Vector<dimension, type, transposed> getRcp() const {
            Vector<dimension, type, transposed> vec(NO_INITIALIZATION);
            for (unsigned i = 0; i < dimension; i++)
                vec[i] = (type)1/element[i];
            return vec;
        };
        /// Returns normalized.
        const Vector<dimension, type, transposed> normalized() const {
            Vector<dimension, type, transposed> vec(NO_INITIALIZATION);
            type f = std::sqrt(SQRLen());
            if (f > type(1e-20))
                f = type(1) / f;
            for (unsigned i = 0; i < dimension; i++)
                vec[i] = element[i] * f;

            return vec;
        };
        /// Normalizes.
        const Vector<dimension, type, transposed>& normalize() {
            type f = std::sqrt(SQRLen());
            if (f > type(1e-20))
                f = type(1) / f;
            for (unsigned i = 0; i < dimension; i++)
                element[i] *= f;

            return *this;
        };
        /// Returns dot-product.
        template<bool otherTransposed>
        const type operator*(const Vector<dimension, type, otherTransposed> &other) const {
            type sum = type(0);
            for (unsigned i = 0; i < dimension; i++)
                sum += element[i] * other[i];

            return sum;
        }
        /// Returns dot-product with self.
        const type SQRLen() const {
            type sum = type(0);
            for (unsigned i = 0; i < dimension; i++)
                sum += element[i] * element[i];

            return sum;
        };
        /// Returns component-wise multiplied.
        const Vector<dimension, type, transposed> operator&(const Vector<dimension, type, transposed> &other) const {
            Vector<dimension, type, transposed> vec(NO_INITIALIZATION);
            for (unsigned i = 0; i < dimension; i++)
                vec[i] = element[i] * other[i];

            return vec;
        };
        /// Converts to std::string, eg: "1.0|-0.5|1.23e-10"
        std::string ToString() const {
            std::stringstream s;
            for (unsigned i = 0; i < dimension; i++) {
                if (i > 0) s << "|";
                s << element[i];
            };
            return s.str();
        };
        /// Converts to std::string, eg: "1.0|-0.5|1.23e-10"
        operator std::string() const {
            return ToString();
        };
        /// Add homogeneous coordinate.
        const Vector<dimension+1, type, transposed> AddHom(type hom) const {
            Vector<dimension+1, type, transposed> vec(Vector<dimension+1, type, transposed>::NO_INITIALIZATION);
            for (unsigned i = 0; i < dimension; i++)
                vec[i] = element[i];
            vec[dimension] = hom;

            return vec;
        };
        /// Remove homogeneous coordinate.
        const Vector<dimension-1, type, transposed> StripHom() const {
            Vector<dimension-1, type, transposed> vec(Vector<dimension-1, type, transposed>::NO_INITIALIZATION);
            for (unsigned i = 0; i < dimension-1; i++)
                vec[i] = element[i];

            return vec;
        };
        const Vector<dimension, type, !transposed> T() const {
            Vector<dimension, type, !transposed> vec(Vector<dimension, type, !transposed>::NO_INITIALIZATION);
            for (unsigned i = 0; i < dimension; i++)
                vec[i] = element[i];
            return vec;
        };

};

/// 2-dim unsigned vector
typedef Vector<2, unsigned> Vector2u;
/// 3-dim unsigned vector
typedef Vector<3, unsigned> Vector3u;

/// 2-dim signed vector
typedef Vector<2, int> Vector2i;
/// 3-dim signed vector
typedef Vector<3, int> Vector3i;

/// 2-dim float vector
typedef Vector<2, float> Vector2f;
/// 3-dim float vector
typedef Vector<3, float> Vector3f;
/// 4-dim float vector
typedef Vector<4, float> Vector4f;

/// 2-dim double vector
typedef Vector<2, double> Vector2d;
/// 3-dim double vector
typedef Vector<3, double> Vector3d;
/// 4-dim double vector
typedef Vector<4, double> Vector4d;

/// Return the cross-product of 2 3-dimensional vectors
template <typename type>
const Vector<3, type> cross(const Vector<3, type> &v1, const Vector<3, type> &v2) {
	Vector<3, type> v(Vector<3, type>::NO_INITIALIZATION);
    v[0] = v1[1]*v2[2] - v1[2]*v2[1];
    v[1] = v1[2]*v2[0] - v1[0]*v2[2];
    v[2] = v1[0]*v2[1] - v1[1]*v2[0];

    return v;
}

///Return 3rd component of the cross-product of 2 2-dimensional vectors embedded in the xy-plane of a 3-dimensional space.
template <typename type>
const type cross(const Vector<2, type> &v1, const Vector<2, type> &v2) {
    return v1[0]*v2[1] - v1[1]*v2[0];
}


/// Convenience function.
template <typename type>
inline Vector<2, type> Fill(type x, type y) {
	Vector<2, type> v(Vector<2, type>::NO_INITIALIZATION);
    v[0] = x;
    v[1] = y;
    return v;
}

/// Convenience function.
template <typename type>
inline Vector<3, type> Fill(type x, type y, type z) {
	Vector<3, type> v(Vector<3, type>::NO_INITIALIZATION);
    v[0] = x;
    v[1] = y;
    v[2] = z;
    return v;
}

/// Convenience function.
template <typename type>
inline Vector<4, type> Fill(type x, type y, type z, type w) {
	Vector<4, type> v(Vector<4, type>::NO_INITIALIZATION);
    v[0] = x;
    v[1] = y;
    v[2] = z;
    v[3] = w;
    return v;
}

inline Vector3f ColorRamp(float f) {
    Vector3f color(Vector3f::NO_INITIALIZATION);
    if (f < 0.0f) {
        color[0] = 0.0f;
        color[1] = 0.0f;
        color[2] = 1.0f;
        return color;
    }
    if (f < 0.25f) {
        color[0] = 0.0f;
        color[1] = f / 0.25f;
        color[2] = 1.0f;
        return color;
    }
    if (f < 0.5f) {
        color[0] = 0.0f;
        color[1] = 1.0f;
        color[2] = (0.5f-f) / 0.25f;
        return color;
    }
    if (f < 0.75f) {
        color[0] = (f-0.5f) / 0.25f;
        color[1] = 1.0f;
        color[2] = 0.0f;
        return color;
    }
    if (f < 1.0f) {
        color[0] = 1.0f;
        color[1] = (1.0f-f) / 0.25f;
        color[2] = 0.0f;
        return color;
    }
    color[0] = 1.0f;
    color[1] = 0.0f;
    color[2] = 0.0f;

    return color;
}


inline Vector<3, unsigned char> clampColor(const Vector3f &color)
{
    Vector<3, unsigned char> clampedColor(Vector<3, unsigned char>::NO_INITIALIZATION);
    clampedColor[0] = (unsigned char) (std::max(0, std::min(255, (int)(color[0] * 255.0f + 0.5f))));
    clampedColor[1] = (unsigned char) (std::max(0, std::min(255, (int)(color[1] * 255.0f + 0.5f))));
    clampedColor[2] = (unsigned char) (std::max(0, std::min(255, (int)(color[2] * 255.0f + 0.5f))));
    return clampedColor;
}

inline uint16_t clampColor565(const Vector3f &color)
{
    Vector<3, unsigned char> clampedColor(Vector<3, unsigned char>::NO_INITIALIZATION);
    clampedColor[0] = (unsigned char) (std::max(0, std::min(31, (int)(color[0] * 31.0f + 0.5f))));
    clampedColor[1] = (unsigned char) (std::max(0, std::min(63, (int)(color[1] * 63.0f + 0.5f))));
    clampedColor[2] = (unsigned char) (std::max(0, std::min(31, (int)(color[2] * 31.0f + 0.5f))));
    return (clampedColor[0] << 0) | (clampedColor[1] << 5) | (clampedColor[2] << 11);
}

/// Convenience function.
inline Vector3f YCoCgToRGB(const Vector3f &YCoCg) {
    Vector3f v(Vector3f::NO_INITIALIZATION);
    v[0] = YCoCg[0] + YCoCg[1] - YCoCg[2];
    v[1] = YCoCg[0] + YCoCg[2];
    v[2] = YCoCg[0] - YCoCg[1] - YCoCg[2];
    return v;
}


template <unsigned dimension, typename type, int epsilonPower>
class VectorComparator
{
    public:
        inline bool operator()(const LinAlg::Vector<dimension, type> &V1, const LinAlg::Vector<dimension, type> &V2) const {
            type ep = (type) pow((type)2.0, epsilonPower);
            for (unsigned i = 0; i < dimension; i++)
                if (V1[i] < V2[i] - ep)
                    return true;
                else
                    if (V1[i] > V2[i] + ep)
                        return false;
            return false;
        }
};

/*
template <int dimension, typename type>
void GramSchmidt(LinAlg::Vector<dimension, type> &V1, LinAlg::Vector<dimension, type> &V2, LinAlg::Vector<dimension, type> &V3)
{
    type u1Sqr = V1*V1;
    if (u1Sqr < 1e-20)
        return;
    type inv_u1Sqr = type(1.0) / u1Sqr;

    V2 = V2 - V1 * ((V2*V1) * inv_u1Sqr);

    type u2Sqr = V2*V2;
    if (u2Sqr < 1e-20)
        return;
    type inv_u2Sqr = type(1.0) / u2Sqr;

    V3 = V3 - V1 * ((V3*V1) * inv_u1Sqr) - V2 * ((V3*V2) * inv_u2Sqr);
}
*/

template <unsigned dimension, typename type>
void GramSchmidt(LinAlg::Vector<dimension, type> &V1, LinAlg::Vector<dimension, type> &V2, LinAlg::Vector<dimension, type> &V3)
{
    V1 *= type(1.0) / std::sqrt(V1.SQRLen());
    V2 = V2 - V1 * (V2*V1);
    V2 *= type(1.0) / std::sqrt(V2.SQRLen());

    V3 = V3 - V1 * (V3*V1);
    V3 = V3 - V2 * (V3*V2);
    V3 *= type(1.0) / std::sqrt(V3.SQRLen());
}


/**
 * @brief Templated implementation of a matrix class.
 * @param Rows The number of rows in the matrix.
 * @param Cols The number of columns in the matrix.
 * @param type The type of the elements.
 * @todo Document this.
 * @todo Write missing methods.
 */
template <unsigned Rows, unsigned Cols, typename type>
class Matrix {
    public:
        /// The Elements are grouped into rows stored as vectors.
        Vector<Cols, type> element[Rows];

        /// Standard constructor, creating an identity matrix.
        Matrix() {
            for (unsigned i = 0; i < Rows; i++)
                for (unsigned j = 0; j < Cols; j++)
                    element[i][j] = type((i == j)?1:0);
        };

        enum NoInitializationEnum {
            NO_INITIALIZATION
        };

        Matrix(NoInitializationEnum) {
        };

        void MakeIdentity() {
            for (unsigned i = 0; i < Rows; i++)
                for (unsigned j = 0; j < Cols; j++)
                    element[i][j] = type((i == j)?1:0);
        }

        /// Access an element (e.g. mat[row][col]).
        Vector<Cols, type>& operator[](unsigned row) {
            return element[row];
        };
        /// Const access to an element (e.g. mat[row][col]).
        const Vector<Cols, type>& operator[](unsigned row) const {
            return element[row];
        };
        /// Add matrix "other".
        Matrix<Rows, Cols, type>& operator+=(const Matrix<Rows, Cols, type> &other) {
            for (unsigned i = 0; i < Rows; i++)
                element[i] += other.element[i];

            return *this;
        };
        /// Subtract matrix "other".
        Matrix<Rows, Cols, type>& operator-=(const Matrix<Rows, Cols, type> &other) {
            for (unsigned i = 0; i < Rows; i++)
                element[i] -= other.element[i];

            return *this;
        };
        /// Multiply matrix with scalar.
        Matrix<Rows, Cols, type>& operator*=(const type f) {
            for (unsigned i = 0; i < Rows; i++)
                element[i] *= f;

            return *this;
        };
        /// Divide matrix by scalar.
        Matrix<Rows, Cols, type>& operator/=(const type f) {
            for (unsigned i = 0; i < Rows; i++)
                element[i] /= f;

            return *this;
        };
        /// Copy matrix.
        Matrix<Rows, Cols, type>& operator=(const Matrix<Rows, Cols, type> &other) {
            for (unsigned i = 0; i < Rows; i++)
                element[i] = other.element[i];

            return *this;
        };
        /// Copy matrix from an array. The array MUST contain Rows*Cols elements.
        Matrix<Rows, Cols, type>& operator=(const type *arr) {
            for (unsigned i = 0; i < Rows; i++)
                element[i] = &arr[i*Cols];

            return *this;
        };
        /// Equal, if all elements are equal.
        bool operator==(const Matrix<Rows, Cols, type> &other) const {
            for (unsigned i = 0; i < Rows; i++)
                if (element[i] != other.element[i])
                    return false;

            return true;
        };
        /// Return sum of both matrices.
        const Matrix<Rows, Cols, type> operator+(const Matrix<Rows, Cols, type> &other) const {
            Matrix<Rows, Cols, type> mat(NO_INITIALIZATION);
            for (unsigned i = 0; i < Rows; i++)
                mat[i] = element[i] + other.element[i];

            return mat;
        };
        /// Return difference of both matrices.
        const Matrix<Rows, Cols, type> operator-(const Matrix<Rows, Cols, type> &other) const {
            Matrix<Rows, Cols, type> mat(NO_INITIALIZATION);
            for (unsigned i = 0; i < Rows; i++)
                mat[i] = element[i] - other.element[i];

            return mat;
        };
        /// Return transposed Matrix.
        const Matrix<Cols, Rows, type> T() const {
            Matrix<Cols, Rows, type> mat(Matrix<Cols, Rows, type>::NO_INITIALIZATION);
            for (unsigned i = 0; i < Rows; i++)
                for (unsigned j = 0; j < Cols; j++)
                    mat[j][i] = element[i][j];

            return mat;
        };
        /// Multiply matrix with scalar and return result.
        const Matrix<Rows, Cols, type> operator*(const type f) const {
            Matrix<Rows, Cols, type> mat(NO_INITIALIZATION);
            for (unsigned i = 0; i < Rows; i++)
                mat[i] = element[i] * f;

            return mat;
        };
        /// Multiply matrix with vector and return result.
        const Vector<Rows, type> operator*(const Vector<Cols, type, false> &vec) const {
            Vector<Rows, type> res(Vector<Rows, type>::NO_INITIALIZATION);
            for (unsigned i = 0; i < Rows; i++)
                res[i] = element[i] * vec;

            return res;
        };
        /// Multiply with other matrix and return result.
        template <unsigned otherCols>
        const Matrix<Rows, otherCols, type> operator*(const Matrix<Cols, otherCols, type> &other) const {
            ///@todo speed up
            Matrix<otherCols, Cols, type> transp(other.T());
            Matrix<Rows, otherCols, type> res(Matrix<Rows, otherCols, type>::NO_INITIALIZATION);
            for (unsigned i = 0; i < Rows; i++)
                for (unsigned j = 0; j < otherCols; j++)
                    res[i][j] = element[i] * transp[j];

            return res;
        }
        const Matrix<Cols-1, Rows-1, type> StripOuterRowsNCols() const {
            Matrix<Cols-1, Rows-1, type> mat(Matrix<Cols-1, Rows-1, type>::NO_INITIALIZATION);
            for (unsigned i = 0; i < Rows-1; i++)
                mat[i] = element[i].StripHom();

            return mat;
        };

		const Matrix<Cols-1, Rows-1, type> MinorMatrix(unsigned skipI, unsigned skipJ) const {
			Matrix<Cols-1, Rows-1, type> minorMatrix(Matrix<Cols-1, Rows-1, type>::NO_INITIALIZATION);
			for (unsigned j = 0; j < skipJ; j++) {
				for (unsigned i = 0; i < skipI; i++)
					minorMatrix[j][i] = (*this)[j][i];
				for (unsigned i = skipI+1; i < Cols; i++)
					minorMatrix[j][i-1] = (*this)[j][i];
			}
			for (unsigned j = skipJ+1; j < Rows; j++) {
				for (unsigned i = 0; i < skipI; i++)
					minorMatrix[j-1][i] = (*this)[j][i];
				for (unsigned i = skipI+1; i < Cols; i++)
					minorMatrix[j-1][i-1] = (*this)[j][i];
			}
			return minorMatrix;
		}

		const Matrix<Rows-1, Cols, type> dropRow(unsigned rowIndex) const {
			Matrix<Rows-1, Cols, type> minorMatrix(Matrix<Rows-1, Cols, type>::NO_INITIALIZATION);
			for (unsigned i = 0; i < rowIndex; i++) {
			    minorMatrix[i] = (*this)[i];
			}
			for (unsigned i = rowIndex+1; i < Cols; i++) {
			    minorMatrix[i-1] = (*this)[i];
			}
			return minorMatrix;
		}

		type cofactor(unsigned i, unsigned j) const {
            if ((i+j)%2 == 0)
                return determinant(MinorMatrix(i, j));
            else
                return -determinant(MinorMatrix(i, j));
		}

		void GaussJordanInvert() {
			if (Cols != Rows)
				throw std::runtime_error("Matrix must be square!");

			Matrix<Cols, Rows, type> inverse;

			type InvScales[Rows];
			for (unsigned i = 0; i < Rows; i++) {
                InvScales[i] = type(1e-20);
                for (unsigned j = 0; j < Cols; j++) {
                    InvScales[i] = std::max(InvScales[i], std::abs((*this)[i][j]));
                }
                InvScales[i] = type(1) / InvScales[i];
			}

			for (unsigned i = 0; i < Cols; i++) {
				unsigned bestRow = i;
				for (unsigned j = i+1; j < Rows; j++) {
					if (std::abs((*this)[j][i] * InvScales[j]) > std::abs((*this)[bestRow][i] * InvScales[bestRow]))
						bestRow = j;
				}
				type pivot = (*this)[bestRow][i];
				if (std::abs(pivot) < type(1e-20))
					throw std::runtime_error("Matrix is singular!");

				(*this)[bestRow] *= (type(1.0) / pivot);
				inverse[bestRow] *= (type(1.0) / pivot);

				Vector<Cols, type> vec2 = inverse[bestRow];
				inverse[bestRow] = inverse[i];
				inverse[i] = vec2;

				Vector<Cols, type> vec1 = (*this)[bestRow];
				(*this)[bestRow] = (*this)[i];
				(*this)[i] = vec1;

				for (unsigned j = 0; j < i; j++) {
					type f = (*this)[j][i];
					(*this)[j] -= vec1 * f;
					inverse[j] -= vec2 * f;
				}
				for (unsigned j = i+1; j < Rows; j++) {
					type f = (*this)[j][i];
					(*this)[j] -= vec1 * f;
					inverse[j] -= vec2 * f;
				}
			}

			*this = inverse;
		}
		void LUDecompose(Matrix<Cols, Rows, type> &L, Matrix<Cols, Rows, type> &U) const {
			if (Cols != Rows)
				throw std::runtime_error("Matrix must be square!");

            U.MakeIdentity();
            L.MakeIdentity();

			for (unsigned i = 0; i < Cols; i++) {

			    for (unsigned n = 0; n < i; n++) {
			        type sum = type(0.0);
                    for (unsigned m = 0; m < n; m++)
                        sum += L[i][m] * U[m][n];
                    L[i][n] = ((*this)[i][n] - sum) / U[n][n];
			    }
			    for (unsigned j = i; j < Rows; j++) {
			        type sum = type(0.0);
                    for (unsigned n = 0; n < i; n++)
                        sum += L[i][n] * U[n][j];
                    U[i][j] = ((*this)[i][j] - sum) / type(1.0);
			    }
			}
		}
		type ComputeDeterminantThroughLUDecomposition() const {
			Matrix<Cols, Rows, type> L(NO_INITIALIZATION);
			Matrix<Cols, Rows, type> U(NO_INITIALIZATION);
		    LUDecompose(L, U);
		    type diag = type(1.0);
		    for (unsigned i = 0; i < Rows; i++)
                diag *= U[i][i];
            return diag;
		}

};

template <unsigned Rows, unsigned Cols, typename type>
const Vector<Rows, type, true> operator*(const Vector<Cols, type, true> &vec, const Matrix<Rows, Cols, type> &mat) {
    Vector<Rows, type, true> res(Vector<Rows, type, true>::NO_INITIALIZATION);
    for (unsigned i = 0; i < Cols; i++) {
        res[i] = type(0.0);
        for (unsigned j = 0; j < Rows; j++)
            res[i] += vec[j] * mat[j][i];
    }
    return res;
}

template <unsigned Rows, unsigned Cols, typename type>
type LaplaceDeterminantExpansion(const Matrix<Rows, Cols, type> &mat) {
    type sum = type(0.0);
    if (Rows > 1)
        for (unsigned j = 0; j < Rows; j++)
            sum += mat[j][Cols-1] * mat.cofactor(Cols-1, j);
    else
        for (unsigned i = 0; i < Cols; i++)
            sum += mat[Rows-1][i] * mat.cofactor(i, Rows-1);
    return sum;
}

template <unsigned Rows, unsigned Cols, typename type>
type determinant(const Matrix<Rows, Cols, type> &mat) {
    return LaplaceDeterminantExpansion(mat);
}

template<typename type>
type determinant(const Matrix<1, 1, type> &mat) {
    return mat[0][0];
}

template<typename type>
type determinant(const Matrix<2, 2, type> &mat) {
    return mat[0][0] * mat[1][1] - mat[1][0] * mat[0][1];
}


template<typename type>
type determinant(const Matrix<3, 3, type> &mat) {
     return mat[0][0] * mat[1][1] * mat[2][2] + mat[0][1] * mat[1][2] * mat[2][0] + mat[0][2] * mat[1][0] * mat[2][1]
              -mat[0][0] * mat[1][2] * mat[2][1] - mat[0][1] * mat[1][0] * mat[2][2] - mat[0][2] * mat[1][1] * mat[2][0];
}

template<unsigned Dim, typename Type>
Matrix<Dim, Dim, Type> DiagonalMatrix(const Vector<Dim, Type> &diagonal)
{
    Matrix<Dim, Dim, Type> M;
    for (unsigned i = 0; i < Dim; i++)
        M[i][i] = diagonal[i];
    return M;
}


/// 2x2 matrix with float elements.
typedef Matrix<2, 2, float> Matrix2x2f;
/// 3x3 matrix with float elements.
typedef Matrix<3, 3, float> Matrix3x3f;
/// 4x4 matrix with float elements.
typedef Matrix<4, 4, float> Matrix4x4f;

typedef Matrix<3, 4, float> Matrix3x4f;
typedef Matrix<3, 4, double> Matrix3x4d;


typedef Matrix<3, 3, double> Matrix3x3d;
typedef Matrix<4, 4, double> Matrix4x4d;

/// Create 2D rotation matrix for use with homogeneous coords.
Matrix3x3f Rotation2D(float angle);
/// Create 2D translation matrix for use with homogeneous coords.
Matrix3x3f Translation2D(const Vector2f &v);
/// Create 2D scaling matrix for use with homogeneous coords.
Matrix3x3f Scale2D(const Vector2f &v);

/// Create 3D translation matrix for use with homogeneous coords.
Matrix4x4f Translation3D(const Vector3f &v);

/// Create 3D translation matrix for use with homogeneous coords.
Matrix4x4f Translation3D(const Vector3f &v);


template<typename Type>
Matrix<3,3,Type> Translation2D(const Vector<2, Type> &v)
{
    Matrix<3,3,Type> m;
    m[0][2] = v[0];
    m[1][2] = v[1];
    return m;
}

template<typename Type>
Matrix<3,3,Type> Scale2D(const Vector<2, Type> &v)
{
    Matrix<3,3,Type> m;
    m[0][0] = v[0];
    m[1][1] = v[1];
    return m;
}

template<typename Type>
Matrix<4,4,Type> Translation3D(const Vector<3, Type> &v)
{
    Matrix<4,4,Type> m;
    m[0][3] = v[0];
    m[1][3] = v[1];
    m[2][3] = v[2];
    return m;
}

template<typename Type>
Matrix<4,4,Type> Scale3D(const Vector<3, Type> &v)
{
    Matrix<4,4,Type> m;
    m[0][0] = v[0];
    m[1][1] = v[1];
    m[2][2] = v[2];
    return m;
}

template<typename Type>
Matrix<4,4,Type> RotateX(Type angle)
{
    Matrix<4,4,Type> m;
    m[1][1] = std::cos(angle);
    m[2][1] = std::sin(angle);
    m[1][2] = -std::sin(angle);
    m[2][2] = std::cos(angle);
    return m;
}

template<typename Type>
Matrix<4,4,Type> RotateY(Type angle)
{
    Matrix<4,4,Type> m;
    m[0][0] = std::cos(angle);
    m[2][0] = std::sin(angle);
    m[0][2] = -std::sin(angle);
    m[2][2] = std::cos(angle);
    return m;
}

template<typename Type>
Matrix<4,4,Type> RotateZ(Type angle)
{
    Matrix<4,4,Type> m;
    m[0][0] = std::cos(angle);
    m[1][0] = std::sin(angle);
    m[0][1] = -std::sin(angle);
    m[1][1] = std::cos(angle);
    return m;
}



template<typename Type>
Matrix<3,3,Type> RotateX3x3(Type angle)
{
    Matrix<3,3,Type> m;
    m[1][1] = std::cos(angle);
    m[2][1] = std::sin(angle);
    m[1][2] = -std::sin(angle);
    m[2][2] = std::cos(angle);
    return m;
}

template<typename Type>
Matrix<3,3,Type> RotateY3x3(Type angle)
{
    Matrix<3,3,Type> m;
    m[0][0] = std::cos(angle);
    m[2][0] = std::sin(angle);
    m[0][2] = -std::sin(angle);
    m[2][2] = std::cos(angle);
    return m;
}

template<typename Type>
Matrix<3,3,Type> RotateZ3x3(Type angle)
{
    Matrix<3,3,Type> m;
    m[0][0] = std::cos(angle);
    m[1][0] = std::sin(angle);
    m[0][1] = -std::sin(angle);
    m[1][1] = std::cos(angle);
    return m;
}


template<typename Type>
Matrix<4,4,Type> ProjectionMatrix(Type fovX, Type aspectRatio, Type zNear, Type zFar)
{
    Type tanFovXHalf = std::tan(fovX / Type(2));
    Type tanFovYHalf = std::tan(fovX / Type(2)) * aspectRatio;

    Type farMinusNear = zFar - zNear;

    Matrix<4,4,Type> m;
    m[0][0] = Type(1.0) / tanFovXHalf;
    m[1][1] = Type(1.0) / tanFovYHalf;
    m[2][2] = -(zFar + zNear) / farMinusNear;
    m[2][3] = Type(-2)*(zFar*zNear) / farMinusNear;
    m[3][2] = Type(-1);
    m[3][3] = Type(0);
    return m;
}



/// Create 3D rotation matrix for use with homogeneous coords.
Matrix4x4f RotateX(float angle);
/// Create 3D rotation matrix for use with homogeneous coords.
Matrix4x4f RotateY(float angle);
/// Create 3D rotation matrix for use with homogeneous coords.
Matrix4x4f RotateZ(float angle);


Matrix4x4f InvertOrientationMatrix(const Matrix4x4f &mat);



template<unsigned Dim, typename Type>
inline Type computeQuadraticFormWithSymmetricMatrix(const Matrix<Dim, Dim, Type> &matrix, const Vector<Dim, Type> &vec)
{
    Type result = Type(0);
    for (unsigned i = 0; i < Dim; i++) {
        Type s = matrix[i][i]*vec[i];
        for (unsigned j = i+1; j < Dim; j++) {
            s += 2*matrix[i][j] * vec[j];
        }
        result += s*vec[i];
    }
    return result;
}

template<>
inline double computeQuadraticFormWithSymmetricMatrix<2, double>(const Matrix<2, 2, double> &matrix, const Vector<2, double> &vec)
{
    double result = double(0);
    {
        double s = matrix[0][0]*vec[0];
        s += 2*matrix[0][1] * vec[1];
        result += s*vec[0];
    }
    {
        double s = matrix[1][1]*vec[0];
        result += s*vec[1];
    }
    return result;
}

template<>
inline double computeQuadraticFormWithSymmetricMatrix<5, double>(const Matrix<5, 5, double> &matrix, const Vector<5, double> &vec)
{
    double result = double(0);
    {
        double s = matrix[0][0]*vec[0];
        s += 2*matrix[0][1] * vec[1];
        s += 2*matrix[0][2] * vec[2];
        s += 2*matrix[0][3] * vec[3];
        s += 2*matrix[0][4] * vec[4];
        result += s*vec[0];
    }
    {
        double s = matrix[1][1]*vec[1];
        s += 2*matrix[1][2] * vec[2];
        s += 2*matrix[1][3] * vec[3];
        s += 2*matrix[1][4] * vec[4];
        result += s*vec[1];
    }
    {
        double s = matrix[2][2]*vec[2];
        s += 2*matrix[2][3] * vec[3];
        s += 2*matrix[2][4] * vec[4];
        result += s*vec[2];
    }
    {
        double s = matrix[3][3]*vec[3];
        s += 2*matrix[3][4] * vec[4];
        result += s*vec[3];
    }
    {
        double s = matrix[4][4]*vec[4];
        result += s*vec[4];
    }
    return result;
}


template<unsigned Dim, typename Type>
inline Type computeQuadraticFormWithNonSymmetricMatrix(const Matrix<Dim, Dim, Type> &matrix, const Vector<Dim, Type> &vec)
{
    Type result = Type(0);
    for (unsigned i = 0; i < Dim; i++) {
        Type s = Type(0);
        for (unsigned j = 0; j < Dim; j++) {
            s += matrix[i][j] * vec[j];
        }
        result += s*vec[i];
    }
    return result;
}

template<>
inline double computeQuadraticFormWithNonSymmetricMatrix<2, double>(const Matrix<2, 2, double> &matrix, const Vector<2, double> &vec)
{
    double result = double(0);
    {
        double s = double(0);
        s += matrix[0][0] * vec[0];
        s += matrix[0][1] * vec[1];
        result += s*vec[0];
    }
    {
        double s = double(0);
        s += matrix[1][0] * vec[0];
        s += matrix[1][1] * vec[1];
        result += s*vec[1];
    }
    return result;
}

template<>
inline double computeQuadraticFormWithNonSymmetricMatrix<5, double>(const Matrix<5, 5, double> &matrix, const Vector<5, double> &vec)
{
    double result = double(0);
    {
        double s = double(0);
        s += matrix[0][0] * vec[0];
        s += matrix[0][1] * vec[1];
        s += matrix[0][2] * vec[2];
        s += matrix[0][3] * vec[3];
        s += matrix[0][4] * vec[4];
        result += s*vec[0];
    }
    {
        double s = double(0);
        s += matrix[1][0] * vec[0];
        s += matrix[1][1] * vec[1];
        s += matrix[1][2] * vec[2];
        s += matrix[1][3] * vec[3];
        s += matrix[1][4] * vec[4];
        result += s*vec[1];
    }
    {
        double s = double(0);
        s += matrix[2][0] * vec[0];
        s += matrix[2][1] * vec[1];
        s += matrix[2][2] * vec[2];
        s += matrix[2][3] * vec[3];
        s += matrix[2][4] * vec[4];
        result += s*vec[2];
    }
    {
        double s = double(0);
        s += matrix[3][0] * vec[0];
        s += matrix[3][1] * vec[1];
        s += matrix[3][2] * vec[2];
        s += matrix[3][3] * vec[3];
        s += matrix[3][4] * vec[4];
        result += s*vec[3];
    }
    {
        double s = double(0);
        s += matrix[4][0] * vec[0];
        s += matrix[4][1] * vec[1];
        s += matrix[4][2] * vec[2];
        s += matrix[4][3] * vec[3];
        s += matrix[4][4] * vec[4];
        result += s*vec[4];
    }
    return result;
}


template <typename type>
class DynamicMatrix {
    private:
        std::vector<type> m_elements;
        unsigned m_rows;
        unsigned m_cols;
    public:
        enum NoInitializationEnum {
            NO_INITIALIZATION
        };

        DynamicMatrix() {
            Resize(0, 0);
        }

        DynamicMatrix(unsigned r, unsigned c) {
            Resize(r, c);
        }

        DynamicMatrix(unsigned r, unsigned c, NoInitializationEnum dummy) {
            Resize(r, c, NO_INITIALIZATION);
        }

        void Resize(unsigned r, unsigned c) {
            m_rows = r;
            m_cols = c;
            m_elements.resize(m_rows*m_cols);
            MakeIdentity();
        }

        void Resize(unsigned r, unsigned c, NoInitializationEnum dummy) {
            m_rows = r;
            m_cols = c;
            m_elements.resize(m_rows*m_cols);
        }

        void MakeIdentity() {
            for (unsigned i = 0; i < getNumRows(); i++)
                for (unsigned j = 0; j < getNumCols(); j++)
                    (*this)(i, j) = type((i == j)?1:0);
        }

        inline type& operator()(unsigned row, unsigned col) {
            return m_elements[col * m_rows + row];
        }

        inline const type& operator()(unsigned row, unsigned col) const {
            return m_elements[col * m_rows + row];
        }

        inline type& operator[](unsigned elem) {
            return m_elements[elem];
        }

        inline const type& operator[](unsigned elem) const {
            return m_elements[elem];
        }


        inline unsigned getNumRows() const { return m_rows; }
        inline unsigned getNumCols() const { return m_cols; }

        DynamicMatrix<type>& operator+=(const DynamicMatrix<type> &other) {
            if (other.getNumRows() != getNumRows())
                throw std::runtime_error("Number of rows do not match!");
            if (other.getNumCols() != getNumCols())
                throw std::runtime_error("Number of rows do not match!");

            for (unsigned i = 0; i < m_elements.size(); i++)
                m_elements[i] += other[i];

            return *this;
        }
        DynamicMatrix<type>& operator-=(const DynamicMatrix<type> &other) {
            if (other.getNumRows() != getNumRows())
                throw std::runtime_error("Number of rows do not match!");
            if (other.getNumCols() != getNumCols())
                throw std::runtime_error("Number of rows do not match!");

            for (unsigned i = 0; i < m_elements.size(); i++)
                m_elements[i] -= other[i];

            return *this;
        }
        /// Multiply matrix with scalar.
        DynamicMatrix<type>& operator*=(const type f) {
            for (unsigned i = 0; i < m_elements.size(); i++)
                m_elements[i] *= f;

            return *this;
        }
        /// Divide matrix by scalar.
        DynamicMatrix<type>& operator/=(const type f) {
            for (unsigned i = 0; i < m_elements.size(); i++)
                m_elements[i] /= f;

            return *this;
        }
        /// Copy matrix.
        DynamicMatrix<type>& operator=(const DynamicMatrix<type> &other) {
            Resize(other.getNumRows(), other.getNumCols());
            for (unsigned i = 0; i < m_elements.size(); i++)
                m_elements[i] = other[i];

            return *this;
        }

        /// Return sum of both matrices.
        const DynamicMatrix<type> operator+(const DynamicMatrix<type> &other) const {
            if (other.getNumRows() != getNumRows())
                throw std::runtime_error("Number of rows do not match!");
            if (other.getNumCols() != getNumCols())
                throw std::runtime_error("Number of rows do not match!");

            DynamicMatrix<type> mat(getNumRows(), getNumCols());
            for (unsigned i = 0; i < m_elements.size(); i++)
                mat[i] = m_elements[i] + other[i];

            return mat;
        }

        /// Return difference of both matrices.
        const DynamicMatrix<type> operator-(const DynamicMatrix<type> &other) const {
            if (other.getNumRows() != getNumRows())
                throw std::runtime_error("Number of rows do not match!");
            if (other.getNumCols() != getNumCols())
                throw std::runtime_error("Number of rows do not match!");

            DynamicMatrix<type> mat(getNumRows(), getNumCols());
            for (unsigned i = 0; i < m_elements.size(); i++)
                mat[i] = m_elements[i] - other[i];

            return mat;
        }

        /// Return transposed Matrix.
        const DynamicMatrix<type> T() const {
            DynamicMatrix<type> mat(getNumCols(), getNumRows());
            for (unsigned i = 0; i < getNumRows(); i++)
                for (unsigned j = 0; j < getNumCols(); j++)
                    mat(j, i) = (*this)(i, j);

            return mat;
        };

        /// Multiply matrix with scalar and return result.
        const DynamicMatrix<type> operator*(const type f) const {
            DynamicMatrix<type> mat(getNumRows(), getNumCols());
            for (unsigned i = 0; i < m_elements.size(); i++)
                mat[i] = m_elements[i] * f;

            return mat;
        };
/*
        /// Multiply matrix with vector and return result.
        const Vector<Rows, type> operator*(const Vector<Cols, type, false> &vec) const {
            Vector<Rows, type> res;
            for (int i = 0; i < Rows; i++)
                res[i] = element[i] * vec;

            return res;
        };
        */
        /// Multiply with other matrix and return result.
        const DynamicMatrix<type> operator*(const DynamicMatrix<type> &other) const {
            if (other.getNumRows() != getNumCols())
                throw std::runtime_error("Number of columns does not match number of rows of second matrix!");

            DynamicMatrix<type> res(getNumRows(), other.getNumCols());
            for (unsigned i = 0; i < getNumRows(); i++)
                for (unsigned j = 0; j < other.getNumCols(); j++) {
                    type sum = (type)0;
                    for (unsigned k = 0; k < getNumCols(); k++)
                        sum += (*this)(i, k) * other(k, j);
                    res(i, j) = sum;
                }

            return res;
        }
/*

		const Matrix<Cols-1, Rows-1, type> MinorMatrix(unsigned skipI, unsigned skipJ) const {
			Matrix<Cols-1, Rows-1, type> minorMatrix;
			for (unsigned j = 0; j < skipJ; j++) {
				for (unsigned i = 0; i < skipI; i++)
					minorMatrix[j][i] = (*this)[j][i];
				for (unsigned i = skipI+1; i < Cols; i++)
					minorMatrix[j][i-1] = (*this)[j][i];
			}
			for (unsigned j = skipJ+1; j < Rows; j++) {
				for (unsigned i = 0; i < skipI; i++)
					minorMatrix[j-1][i] = (*this)[j][i];
				for (unsigned i = skipI+1; i < Cols; i++)
					minorMatrix[j-1][i-1] = (*this)[j][i];
			}
			return minorMatrix;
		}

		type cofactor(unsigned i, unsigned j) const {
            if ((i+j)%2 == 0)
                return determinant(MinorMatrix(i, j));
            else
                return -determinant(MinorMatrix(i, j));
		}
*/
		void GaussJordanInvert() {
			if (getNumCols() != getNumRows())
				throw std::runtime_error("Matrix must be square!");

			DynamicMatrix<type> inverse(getNumRows(), getNumCols());

			type InvScales[getNumRows()];
			for (unsigned i = 0; i < getNumRows(); i++) {
                InvScales[i] = type(1e-20);
                for (unsigned j = 0; j < getNumCols(); j++) {
                    InvScales[i] = std::max(InvScales[i], std::abs((*this)(i,j)));
                }
                InvScales[i] = 1.0 / InvScales[i];
			}

			for (unsigned i = 0; i < getNumCols(); i++) {
				unsigned bestRow = i;
				for (unsigned j = i+1; j < getNumRows(); j++) {
					if (std::abs((*this)(j, i) * InvScales[j]) > std::abs((*this)(bestRow, i) * InvScales[bestRow]))
						bestRow = j;
				}
				type pivot = (*this)(bestRow, i);
				if (std::abs(pivot) < 1e-20)
					throw std::runtime_error("Matrix is singular!");

                for (unsigned j = 0; j < getNumCols(); j++) {
                    (*this)(bestRow, j) *= (1.0 / pivot);
                    inverse(bestRow, j) *= (1.0 / pivot);
                }

                for (unsigned j = 0; j < getNumCols(); j++) {
                    type h = inverse(bestRow, j);
                    inverse(bestRow, j) = inverse(i, j);
                    inverse(i, j) = h;

                    h = (*this)(bestRow, j);
                    (*this)(bestRow, j) = (*this)(i, j);
                    (*this)(i, j) = h;
                }

				for (unsigned j = 0; j < i; j++) {
					type f = (*this)(j, i);
                    for (unsigned k = 0; k < getNumCols(); k++) {
                        (*this)(j, k) -= (*this)(i, k) * f;
                        inverse(j, k) -= inverse(i, k) * f;
                    }
				}
				for (unsigned j = i+1; j < getNumRows(); j++) {
					type f = (*this)(j, i);
                    for (unsigned k = 0; k < getNumCols(); k++) {
                        (*this)(j, k) -= (*this)(i, k) * f;
                        inverse(j, k) -= inverse(i, k) * f;
                    }
				}
			}

			*this = inverse;
		}

		void LUDecompose(DynamicMatrix<type> &L, DynamicMatrix<type> &U) const {
			if (getNumCols() != getNumRows())
				throw std::runtime_error("Matrix must be square!");

            U.Resize(getNumCols(), getNumRows());
            L.Resize(getNumCols(), getNumRows());

			for (unsigned i = 0; i < getNumCols(); i++) {

			    for (unsigned n = 0; n < i; n++) {
			        type sum = 0.0;
                    for (unsigned m = 0; m < n; m++)
                        sum += L(i, m) * U(m, n);
                    L(i, n) = ((*this)(i, n) - sum) / U(n, n);
			    }
			    for (unsigned j = i; j < getNumRows(); j++) {
			        type sum = 0.0;
                    for (unsigned n = 0; n < i; n++)
                        sum += L(i, n) * U(n, j);
                    U(i, j) = ((*this)(i, j) - sum) / 1.0;
			    }
			}
		}

		type ComputeDeterminantThroughLUDecomposition() const {
		    DynamicMatrix<type> L;
		    DynamicMatrix<type> U;
		    LUDecompose(L, U);
		    type diag = 1.0;
		    for (unsigned i = 0; i < getNumRows(); i++)
                diag *= U(i, i);
            return diag;
		}

};


/**
 *  @}
 */

}


#endif
