/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef PCVTOOLBOX_HPP_INCLUDED
#define PCVTOOLBOX_HPP_INCLUDED

/**
 * @file
 * @author Andreas Ley
 */

#include "LinAlg.h"
#include "SVD.hpp"
#include "SSEMath.h"
#include <smmintrin.h>

#include "AlignedAllocator.h"

//#include "Camera.hpp"
#include <vector>
#include <string.h>

#include "RasterImage.h"
#include <iomanip>

template<unsigned Rows, unsigned Cols, typename Type>
std::ostream &output(std::ostream &stream, const LinAlg::Matrix<Rows, Cols, Type> &m)
{
    stream << "[ " << std::endl;
    for (unsigned i = 0; i < Rows; i++) {
        for (unsigned j = 0; j < Cols; j++)
            stream << " " << std::setw(15) << m[i][j];
        if (i < Rows-1)
            stream << " ; " << std::endl;
    }
    stream << "]" << std::endl;
    return stream;
}

std::ostream &operator<<(std::ostream &stream, const LinAlg::Matrix<3, 3, double> &m);
std::ostream &operator<<(std::ostream &stream, const LinAlg::Matrix<3, 4, double> &m);

std::ostream &operator<<(std::ostream &stream, const LinAlg::Matrix<4, 4, double> &m);
std::ostream &operator<<(std::ostream &stream, const LinAlg::Matrix<16, 16, double> &m);

std::ostream &operator<<(std::ostream &stream, const LinAlg::Matrix<3, 3, float> &m);
std::ostream &operator<<(std::ostream &stream, const LinAlg::Matrix<3, 4, float> &m);

std::ostream &operator<<(std::ostream &stream, const LinAlg::Matrix<4, 4, float> &m);

std::ostream &operator<<(std::ostream &stream, const LinAlg::Matrix<6, 6, float> &m);

std::ostream &operator<<(std::ostream &stream, const LinAlg::Matrix<16, 16, float> &m);

namespace PCV {

template<typename Type>
LinAlg::Matrix<4, 4, Type> computeInverseProjectionMatrix(const LinAlg::Matrix<3, 4, Type> &P)
{
    LinAlg::Matrix<4, 4, Type> subMatrix;
    for (unsigned i = 0; i < 3; i++)
        for (unsigned j = 0; j < 4; j++)
            subMatrix[i][j] = P[i][j];

    subMatrix.GaussJordanInvert();
    return subMatrix;
}

template<typename Type>
void getFrustrumCornersFromProjectionMatrix(const LinAlg::Matrix<3, 4, Type> &P, LinAlg::Vector<3, Type> *cornerPoints)
{
    LinAlg::Matrix<4, 4, Type> subMatrix = computeInverseProjectionMatrix(P);

    for (unsigned i = 0; i < 4; i++) {
        LinAlg::Vector<4, Type> p;
        p[0] = ((i & 1) == 0)?-1.0:1.0;
        p[1] = ((i & 2) == 0)?-1.0:1.0;
        p[2] = 0.1;
        p[0] *= p[2];
        p[1] *= p[2];
        p[3] = 1.0;

        p = subMatrix * p;

        cornerPoints[i] = p.StripHom();
    }
}

template<unsigned Dim, typename Type>
void getPositionOfClosestProximityBetweenLines(const LinAlg::Vector<Dim, Type> &S1, const LinAlg::Vector<Dim, Type> &D1,
                                               const LinAlg::Vector<Dim, Type> &S2, const LinAlg::Vector<Dim, Type> &D2,
                                               Type &lambda1, Type &lambda2)
{
    LinAlg::Matrix<2, 2, Type> M;
    M[0][0] = D1 * D1;
    M[0][1] = -(D1*D2);
    M[1][0] = -M[0][1];
    M[1][1] = -(D2 * D2);

    Type det = LinAlg::determinant(M);
    if (fabs(det) < (Type)1e-10) {
        lambda1 = (Type)0.0;
        lambda2 = (Type)0.0;
        return;
    }

    Type rcpDet = ((Type)1) / det;

    LinAlg::Matrix<2, 2, Type> InvM;
    InvM[0][0] = M[1][1] * rcpDet;
    InvM[0][1] = -M[0][1] * rcpDet;
    InvM[1][0] = -M[1][0] * rcpDet;
    InvM[1][1] = M[0][0] * rcpDet;

    LinAlg::Vector<2, Type> inHom;
    inHom[0] = D1*S2 - D1*S1;
    inHom[1] = D2*S2 - D2*S1;


    LinAlg::Vector<2, Type> lambdas = InvM * inHom;
    lambda1 = lambdas[0];
    lambda2 = lambdas[1];
}


template<typename Type>
void drawEpipolarLine(RasterImage &image, const LinAlg::Vector<3, Type> &epipolarLine, uint32_t color)
{
    LinAlg::Vector<3, Type> borders[4];

    borders[0] = LinAlg::cross(epipolarLine, LinAlg::Fill<Type>(1.0f, 0.0f, 0.0f));
    borders[1] = LinAlg::cross(epipolarLine, LinAlg::Fill<Type>(1.0f, 0.0f, -(int)image.getWidth()));
    borders[2] = LinAlg::cross(epipolarLine, LinAlg::Fill<Type>(0.0f, 1.0f, 0.0f));
    borders[3] = LinAlg::cross(epipolarLine, LinAlg::Fill<Type>(0.0f, 1.0f, -(int)image.getHeight()));

    LinAlg::Vector2i start, end;

    unsigned index;
    for (index = 0; index < 4; index++) {
        if (fabs(borders[index][2]) > 1e-20) {
            LinAlg::Vector2i p;
            p[0] = (int)(borders[index][0] / borders[index][2]);
            p[1] = (int)(borders[index][1] / borders[index][2]);

            if ((p[0] > -5) && (p[0] < (int)image.getWidth()+5) &&
                (p[1] > -5) && (p[1] < (int)image.getHeight()+5)) {

                start = p;
                break;
            }
        }
    }
    for (index = index+1; index < 4; index++) {
        if (fabs(borders[index][2]) > 1e-20) {
            LinAlg::Vector2i p;
            p[0] = (int)(borders[index][0] / borders[index][2]);
            p[1] = (int)(borders[index][1] / borders[index][2]);

            if ((p[0] > -5) && (p[0] < (int)image.getWidth()+5) &&
                (p[1] > -5) && (p[1] < (int)image.getHeight()+5)) {

                end = p;
                break;
            }
        }
    }
    image.drawLine(start, end, color);

}



template<typename Type>
void forceSingularity(LinAlg::Matrix<3, 3, Type> &f)
{
    LinAlg::Matrix<3, 3, Type> FtF = f.T() * f;

    LinAlg::Matrix<3, 3, Type> rightSVecs;
    LinAlg::Vector<3, Type> rightSVals;

    LinAlg::computeEigenValues<3, Type>(FtF, rightSVals, &rightSVecs);

    rightSVals[0] = sqrtf(std::max<Type>(0.0, rightSVals[0]));
    rightSVals[1] = sqrtf(std::max<Type>(0.0, rightSVals[1]));
    rightSVals[2] = sqrtf(std::max<Type>(0.0, rightSVals[2]));

    unsigned smallestSV = 0;
    for (unsigned i = 1; i < 3; i++) {
        if (rightSVals[i] < rightSVals[smallestSV])
            smallestSV = i;
    }

    if (rightSVals[smallestSV] == 0.0)
        return;

    LinAlg::Matrix<3, 3, Type> leftSVecs = f * rightSVecs * LinAlg::DiagonalMatrix<3, Type>(LinAlg::Fill<Type>(1.0 / rightSVals[0], 1.0 / rightSVals[1], 1.0 / rightSVals[2]));

    rightSVals[smallestSV] = 0.0;


    f = leftSVecs * LinAlg::DiagonalMatrix<3, Type>(rightSVals) * rightSVecs.T();
}


template<unsigned dimension, typename Type>
void transform(const LinAlg::Matrix<dimension+1, dimension+1, Type> &matrix, const std::vector<LinAlg::Vector<dimension+1, Type> > &srcPoints, std::vector<LinAlg::Vector<dimension+1, Type> > &dstPoints)
{
    dstPoints.resize(srcPoints.size());
    for (unsigned i = 0; i < srcPoints.size(); i++)
        dstPoints[i] = matrix * srcPoints[i];
}


template<unsigned dimension, typename Type>
LinAlg::Matrix<dimension+1, dimension+1, Type> getCondition(const std::vector<LinAlg::Vector<dimension+1, Type> > &points)
{
    LinAlg::Vector<dimension, Type> translation;
    for (unsigned i = 0; i < points.size(); i++) {
        translation += points[i].StripHom() / points[i][dimension];
    }
    translation /= points.size();


    LinAlg::Vector<dimension, Type> scale;
    for (unsigned i = 0; i < points.size(); i++)
        for (unsigned j = 0; j < dimension; j++)
            scale[j] += fabs(points[i][j] / points[i][dimension] - translation[j]);
    scale /= points.size();

    LinAlg::Vector<dimension, Type> factor;
    for (unsigned i = 0; i < dimension; i++)
        factor[i] = std::abs(scale[i]) < 1e-50?1.0:1.0/scale[i];

    LinAlg::Matrix<dimension+1, dimension+1, Type> conditioningMatrix;

    for (unsigned i = 0; i < dimension; i++) {
        conditioningMatrix[i][i] = factor[i];
        conditioningMatrix[i][dimension] = -translation[i] * factor[i];
    }


    return conditioningMatrix;
}

template<typename Type>
std::vector<LinAlg::Vector<9, Type> > getDesignMatrix_fundamental(const std::vector<LinAlg::Vector<3, Type> > &first, const std::vector<LinAlg::Vector<3, Type> > &second)
{
    if (first.size() != second.size())
        throw std::runtime_error("wrong number of points!");

    std::vector<LinAlg::Vector<9, Type> > A;
    A.resize(std::max<unsigned>(9, first.size()));
    for (unsigned i = 0; i < first.size(); i++) {
        A[i][0] = first[i][0]*second[i][0];
        A[i][1] = first[i][1]*second[i][0];
        A[i][2] = first[i][2]*second[i][0];
        A[i][3] = first[i][0]*second[i][1];
        A[i][4] = first[i][1]*second[i][1];
        A[i][5] = first[i][2]*second[i][1];
        A[i][6] = first[i][0]*second[i][2];
        A[i][7] = first[i][1]*second[i][2];
        A[i][8] = first[i][2]*second[i][2];
    }
    return A;
}

template<typename Type>
void decondition_fundamental(const LinAlg::Matrix<3, 3, Type> &T_1, const LinAlg::Matrix<3, 3, Type> &T_2, LinAlg::Matrix<3, 3, Type> &fundamental)
{
    fundamental = T_2.T() * fundamental * T_1;
}

template<typename Type>
LinAlg::Matrix<3, 3, Type> getFundamentalMatrix(const std::vector<LinAlg::Vector<3, Type> > &first, const std::vector<LinAlg::Vector<3, Type> > &second)
{
    LinAlg::Matrix<3, 3, Type> T_1 = getCondition<2>(first);
    LinAlg::Matrix<3, 3, Type> T_2 = getCondition<2>(second);

    std::vector<LinAlg::Vector<3, Type> > conditionedPoints1;
    std::vector<LinAlg::Vector<3, Type> > conditionedPoints2;

    transform<2>(T_1, first, conditionedPoints1);
    transform<2>(T_2, second, conditionedPoints2);


    std::vector<LinAlg::Vector<9, Type> > designMatrix = getDesignMatrix_fundamental(conditionedPoints1, conditionedPoints2);

    LinAlg::Vector<9, Type> singularValues;
    LinAlg::Matrix<9, 9, Type> singularVectors;

    try {
        LinAlg::SVD::computeRightSingularVectorsForRealMatrix<std::vector<LinAlg::Vector<9, Type> >, 9, Type>(designMatrix, singularValues, singularVectors);
    } catch(...) {
        /*
        std::cout << "Design = [ ";
        for (unsigned i = 0; i < first.size(); i++) {
            for (unsigned j = 0; j < 9; j++)
                std::cout << " " << designMatrix[i][j];
            if (i+1 < first.size())
                std::cout << ";" << std::endl;
        }
        std::cout << "];" << std::endl;

        output<3,3,Type>(std::cout, T_1);
        output<3,3,Type>(std::cout, T_2);
        for (unsigned i = 0; i < std::min<unsigned>(first.size(), 10u); i++) {
            std::cout << i << ": " << (std::string) first[i] << " " << (std::string) second[i] << std::endl;
            std::cout << i << ": " << (std::string) conditionedPoints1[i] << " " << (std::string) conditionedPoints2[i] << std::endl;
        }
        */
        throw;
    }

    unsigned smallestSV = 0;
    for (unsigned i = 1; i < 9; i++) {
        if (singularValues[i] < singularValues[smallestSV])
            smallestSV = i;
    }


    LinAlg::Vector<9, Type> fundamentalAsVector = singularVectors.T()[smallestSV];

    LinAlg::Matrix<3, 3, Type> fundamental;
    memcpy(&fundamental[0][0], &fundamentalAsVector[0], 9*sizeof(Type));

    forceSingularity<Type>(fundamental);

    decondition_fundamental<Type>(T_1, T_2, fundamental);

    return fundamental;
}

template<typename Type>
Type getError(const std::vector<LinAlg::Vector<3, Type> > &first, const std::vector<LinAlg::Vector<3, Type> > &second, const LinAlg::Matrix<3, 3, Type> &fundamental, Type exp)
{

    LinAlg::Matrix<3, 3, Type> fundamentalT = fundamental.T();

    Type sum = 0.0f;

    for (unsigned i = 0; i < first.size(); i++) {
        LinAlg::Vector<3, Type> Fx = fundamental * (first[i] / first[i][2]);
        LinAlg::Vector<3, Type> FTxdash = fundamentalT * (second[i] / second[i][2]);

        Type f = Fx * (second[i] / second[i][2]);
        Type denom = Fx.StripHom().SQRLen() + FTxdash.StripHom().SQRLen();
        if (denom > 0.0) {
            sum += powf(f*f/denom, exp);
        }
    }
    return sum / first.size();
}


template<typename Type>
void getErrors(const std::vector<LinAlg::Vector<3, Type> > &first, const std::vector<LinAlg::Vector<3, Type> > &second, const LinAlg::Matrix<3, 3, Type> &fundamental, Type exp,
              std::vector<Type> &errors)
{

    LinAlg::Matrix<3, 3, Type> fundamentalT = fundamental.T();

    for (unsigned i = 0; i < first.size(); i++) {
        LinAlg::Vector<3, Type> Fx = fundamental * (first[i] / first[i][2]);
        LinAlg::Vector<3, Type> FTxdash = fundamentalT * (second[i] / second[i][2]);

        Type f = Fx * (second[i] / second[i][2]);
        Type denom = Fx.StripHom().SQRLen() + FTxdash.StripHom().SQRLen();
        if (denom > 0.0) {
            errors[i] = powf(f*f/denom, exp);
        } else
            errors[i] = -1.0f;
    }
}


template<>
float getError(const std::vector<LinAlg::Vector<3, float> > &first, const std::vector<LinAlg::Vector<3, float> > &second, const LinAlg::Matrix<3, 3, float> &fundamental, float exp);


template<typename Type>
LinAlg::Vector<3, Type> getEpipole(const LinAlg::Matrix<3, 3, Type> &fundamental)
{
    LinAlg::Vector<3, Type> SVals;
    LinAlg::Matrix<3, 3, Type> SVecs;

    LinAlg::SVD::computeRightSingularVectorsForRealMatrixFixedRows<3, 3, Type>(fundamental, SVals, SVecs);

    unsigned smallest = 0;
    for (unsigned i = 1; i < 3; i++)
        if (SVals[i] < SVals[smallest])
            smallest = i;

    return SVecs.T()[smallest];
}

template<typename Type>
LinAlg::Matrix<3, 3, Type> constructSkewSymmetric(const LinAlg::Vector<3, Type> &vec)
{
    LinAlg::Matrix<3, 3, Type> result;

    result[0][0] = Type(0);
    result[0][1] = -vec[2];
    result[0][2] = vec[1];

    result[1][0] = vec[2];
    result[1][1] = Type(0);
    result[1][2] = -vec[0];

    result[2][0] = -vec[1];
    result[2][1] = vec[0];
    result[2][2] = Type(0);

    return result;
}


template<typename Type>
LinAlg::Vector<4, Type> triangulatePoint(const LinAlg::Matrix<3, 4, Type> &P1, const LinAlg::Vector<3, Type> &U1, const LinAlg::Matrix<3, 4, Type> &P2, const LinAlg::Vector<3, Type> &U2)
{
    LinAlg::Matrix<4, 4, Type> designMatrix;


    for (unsigned i = 0; i < 4; i++)
        for (unsigned j = 0; j < 4; j++)
            designMatrix[i][j] = 0.0;

    for (unsigned i = 0; i < 4; i++)
        designMatrix[0][i] = U1[0] * P1[2][i] - U1[2] * P1[0][i];

    for (unsigned i = 0; i < 4; i++)
        designMatrix[1][i] = U1[1] * P1[2][i] - U1[2] * P1[1][i];

    for (unsigned i = 0; i < 4; i++)
        designMatrix[2][i] = U2[0] * P2[2][i] - U2[2] * P2[0][i];

    for (unsigned i = 0; i < 4; i++)
        designMatrix[3][i] = U2[1] * P2[2][i] - U2[2] * P2[1][i];

/*
    designMatrix[2][2] = 1.0;
    designMatrix[2][3] = -1.0;


    std::cout << "D = [ " << std::endl;
    for (unsigned i = 0; i < 4; i++) {
        for (unsigned j = 0; j < 4; j++)
            std::cout << " " << designMatrix[i][j];
        if (i < 2)
            std::cout << " ; " << std::endl;
    }
    std::cout << "]" << std::endl;

    std::cout << "P1 = [ " << std::endl;
    for (unsigned i = 0; i < 3; i++) {
        for (unsigned j = 0; j < 4; j++)
            std::cout << " " << P1[i][j];
        if (i < 2)
            std::cout << " ; " << std::endl;
    }
    std::cout << "]" << std::endl;

    std::cout << "U1 = [ " << U1[0] << " ; " << U1[1] << " ; " << U1[2] << " ] " << std::endl;
*/

    LinAlg::Vector<4, Type> SVals;
    LinAlg::Matrix<4, 4, Type> SVecs;

    LinAlg::SVD::computeRightSingularVectorsForRealMatrixFixedRows<4, 4, Type>(designMatrix, SVals, SVecs);

    unsigned smallest = 0;
    for (unsigned i = 1; i < 4; i++)
        if (SVals[i] < SVals[smallest])
            smallest = i;

    LinAlg::Vector<4, Type> position = SVecs.T()[smallest];
/*
    if (position[2]*position[3] < 0) {
        std::cout << "P1 = " << P1 << std::endl;
        std::cout << "U1: " << (std::string) U1 << std::endl;
        std::cout << "P2 = " << P2 << std::endl;
        std::cout << "U2: " << (std::string) U1 << std::endl;
        std::cout << "D = [ " << std::endl;
        for (unsigned i = 0; i < 4; i++) {
            for (unsigned j = 0; j < 4; j++)
                std::cout << " " << designMatrix[i][j];
            if (i < 3)
                std::cout << " ; " << std::endl;
        }
        std::cout << "]" << std::endl;
        exit(1);

    }

*/

    position /= position[3];

    return position;
}

template<typename Type>
void refinePointTriangulation(const LinAlg::Matrix<3, 4, Type> &P1, const LinAlg::Vector<3, Type> &U1, const LinAlg::Matrix<3, 4, Type> &P2, const LinAlg::Vector<3, Type> &U2,
            LinAlg::Vector<4, Type> &position)
{
    Type lasterror = 1000.0;
    Type stepsize = 1.0;
    for (unsigned iterations = 0; iterations < 10000; iterations++) {
        LinAlg::Vector<3, Type> gradient;
        Type error = 0.0;
        {
            const LinAlg::Matrix<3,4,Type> &P = P1;
            const LinAlg::Vector<3, Type> &y = U1;

            LinAlg::Vector<3, Type> z = P * position;
            LinAlg::Vector<2, Type> zNorm = z.StripHom() / z[2];

            LinAlg::Vector<2, Type> d = (zNorm - y.StripHom());
            error += d.SQRLen();

            gradient[0] += (P[0][0] - P[2][0] * zNorm[0]) / z[2] * d[0] +
                           (P[1][0] - P[2][0] * zNorm[1]) / z[2] * d[1];

            gradient[1] += (P[0][1] - P[2][1] * zNorm[0]) / z[2] * d[0] +
                           (P[1][1] - P[2][1] * zNorm[1]) / z[2] * d[1];

            gradient[2] += (P[0][2] - P[2][2] * zNorm[0]) / z[2] * d[0] +
                           (P[1][2] - P[2][2] * zNorm[1]) / z[2] * d[1];
        }
        {
            const LinAlg::Matrix<3,4,Type> &P = P2;
            const LinAlg::Vector<3, Type> &y = U2;

            LinAlg::Vector<3, Type> z = P * position;
            LinAlg::Vector<2, Type> zNorm = z.StripHom() / z[2];

            LinAlg::Vector<2, Type> d = (zNorm - y.StripHom());
            error += d.SQRLen();

            gradient[0] += (P[0][0] - P[2][0] * zNorm[0]) / z[2] * d[0] +
                           (P[1][0] - P[2][0] * zNorm[1]) / z[2] * d[1];

            gradient[1] += (P[0][1] - P[2][1] * zNorm[0]) / z[2] * d[0] +
                           (P[1][1] - P[2][1] * zNorm[1]) / z[2] * d[1];

            gradient[2] += (P[0][2] - P[2][2] * zNorm[0]) / z[2] * d[0] +
                           (P[1][2] - P[2][2] * zNorm[1]) / z[2] * d[1];
        }

        if (error <= lasterror) {
            stepsize *= 1.2;
        } else {
            stepsize *= 0.5;
        }
        lasterror = error;

        position = (position.StripHom() - gradient * stepsize).AddHom(1.0);
    }
}

template<typename Type>
std::vector<LinAlg::Vector<16, Type> > getDesignMatrix_homography3D(const std::vector<LinAlg::Vector<4, Type> > &first, const std::vector<LinAlg::Vector<4, Type> > &second)
{
    if (first.size() != second.size())
        throw std::runtime_error("wrong number of points!");


/*
    Y.x = H[0] . X
    Y.y = H[1] . X
    Y.z = H[2] . X
    Y.w = H[3] . X

    Y.x / Y.w = H[0] . X / H[3] . X
    Y.y / Y.w = H[1] . X / H[3] . X
    Y.z / Y.w = H[2] . X / H[3] . X


    Y.x * H[3] . X = H[0] . X * Y.w
    Y.y * H[3] . X = H[1] . X * Y.w
    Y.z * H[3] . X = H[2] . X * Y.w


    Y.x * H[3] . X - H[0] . X * Y.w = 0
    Y.y * H[3] . X - H[1] . X * Y.w = 0
    Y.z * H[3] . X - H[2] . X * Y.w = 0
*/

    std::vector<LinAlg::Vector<16, Type> > A;
    A.resize(std::max<unsigned>(16, first.size()*3));
    for (unsigned i = 0; i < first.size(); i++) {

        A[i*3+0][0] = -first[i][3]*second[i][0];
        A[i*3+0][1] = -first[i][3]*second[i][1];
        A[i*3+0][2] = -first[i][3]*second[i][2];
        A[i*3+0][3] = -first[i][3]*second[i][3];

        A[i*3+0][12] = first[i][0]*second[i][0];
        A[i*3+0][13] = first[i][0]*second[i][1];
        A[i*3+0][14] = first[i][0]*second[i][2];
        A[i*3+0][15] = first[i][0]*second[i][3];


        A[i*3+1][4] = -first[i][3]*second[i][0];
        A[i*3+1][5] = -first[i][3]*second[i][1];
        A[i*3+1][6] = -first[i][3]*second[i][2];
        A[i*3+1][7] = -first[i][3]*second[i][3];

        A[i*3+1][12] = first[i][1]*second[i][0];
        A[i*3+1][13] = first[i][1]*second[i][1];
        A[i*3+1][14] = first[i][1]*second[i][2];
        A[i*3+1][15] = first[i][1]*second[i][3];


        A[i*3+2][8] = -first[i][3]*second[i][0];
        A[i*3+2][9] = -first[i][3]*second[i][1];
        A[i*3+2][10] = -first[i][3]*second[i][2];
        A[i*3+2][11] = -first[i][3]*second[i][3];

        A[i*3+2][12] = first[i][2]*second[i][0];
        A[i*3+2][13] = first[i][2]*second[i][1];
        A[i*3+2][14] = first[i][2]*second[i][2];
        A[i*3+2][15] = first[i][2]*second[i][3];
    }
    return A;
}

template<typename Type>
void decondition(const LinAlg::Matrix<4, 4, Type> &T_1, const LinAlg::Matrix<4, 4, Type> &T_2, LinAlg::Matrix<4, 4, Type> &homography)
{
    LinAlg::Matrix<4, 4, Type> T_1_Inv = T_1;
    T_1_Inv.GaussJordanInvert();

    homography = T_1_Inv * homography * T_2;
}


template<typename Type>
LinAlg::Matrix<4, 4, Type> homography3D(const std::vector<LinAlg::Vector<4, Type> > &points1, const std::vector<LinAlg::Vector<4, Type> > &points2)
{
    LinAlg::Matrix<4, 4, Type> conditioning1 = getCondition<3>(points1);
    LinAlg::Matrix<4, 4, Type> conditioning2 = getCondition<3>(points2);

    std::vector<LinAlg::Vector<4, Type> > conditionedPoints1, conditionedPoints2;

    transform<3>(conditioning1, points1, conditionedPoints1);
    transform<3>(conditioning2, points2, conditionedPoints2);

    std::vector<LinAlg::Vector<16, Type> > designMatrix = getDesignMatrix_homography3D(conditionedPoints1, conditionedPoints2);
/*
    std::cout << "D = [ " << std::endl;
    for (unsigned i = 0; i < designMatrix.size(); i++) {
        for (unsigned j = 0; j < 16; j++)
            std::cout << " " << designMatrix[i][j];
        if (i < designMatrix.size()-1)
            std::cout << " ; " << std::endl;
    }
    std::cout << "]" << std::endl;
*/

    LinAlg::Vector<16, Type> singularValues;
    LinAlg::Matrix<16, 16, Type> singularVectors;

    LinAlg::SVD::computeRightSingularVectorsForRealMatrix<std::vector<LinAlg::Vector<16, Type> >, 16, Type>(designMatrix, singularValues, singularVectors);
/*
    std::cout << "SVals: " << (std::string) singularValues << std::endl;
    std::cout << "SVecs: " << singularVectors << std::endl;
*/
    unsigned smallestSV = 0;
    for (unsigned i = 1; i < 16; i++) {
        if (singularValues[i] < singularValues[smallestSV])
            smallestSV = i;
    }


    LinAlg::Vector<16, Type> homographyAsVector = singularVectors.T()[smallestSV];

    LinAlg::Matrix<4, 4, Type> homography;
    memcpy(&homography[0][0], &homographyAsVector[0], 16*sizeof(Type));


    decondition<Type>(conditioning1, conditioning2, homography);

    return homography;
}

template<typename Type>
void decondition(const LinAlg::Matrix<3, 3, Type> &T_1, const LinAlg::Matrix<4, 4, Type> &T_2, LinAlg::Matrix<3, 4, Type> &projectionMatrix)
{
    LinAlg::Matrix<3, 3, Type> T_1_Inv = T_1;
    T_1_Inv.GaussJordanInvert();

    projectionMatrix = T_1_Inv * projectionMatrix * T_2;
}

template<typename Type, class Alloc>
std::vector<LinAlg::Vector<12, Type>, Alloc> getDesignMatrix_projectionMatrixFromWSPoints(const std::vector<LinAlg::Vector<3, Type> > &points2D, const std::vector<LinAlg::Vector<4, Type> > &points3D)
{
    if (points2D.size() != points3D.size())
        throw std::runtime_error("wrong number of points!");

    std::vector<LinAlg::Vector<12, Type>, Alloc> A;
    A.resize(std::max<unsigned>(12, points2D.size()*2));
    for (unsigned i = 0; i < points2D.size(); i++) {
        A[i*2+0][0] = -points2D[i][2] * points3D[i][0];
        A[i*2+0][1] = -points2D[i][2] * points3D[i][1];
        A[i*2+0][2] = -points2D[i][2] * points3D[i][2];
        A[i*2+0][3] = -points2D[i][2] * points3D[i][3];

        A[i*2+0][8] = points2D[i][0] * points3D[i][0];
        A[i*2+0][9] = points2D[i][0] * points3D[i][1];
        A[i*2+0][10] = points2D[i][0] * points3D[i][2];
        A[i*2+0][11] = points2D[i][0] * points3D[i][3];

        A[i*2+1][4] = -points2D[i][2] * points3D[i][0];
        A[i*2+1][5] = -points2D[i][2] * points3D[i][1];
        A[i*2+1][6] = -points2D[i][2] * points3D[i][2];
        A[i*2+1][7] = -points2D[i][2] * points3D[i][3];

        A[i*2+1][8] = points2D[i][1] * points3D[i][0];
        A[i*2+1][9] = points2D[i][1] * points3D[i][1];
        A[i*2+1][10] = points2D[i][1] * points3D[i][2];
        A[i*2+1][11] = points2D[i][1] * points3D[i][3];
    }
    return A;

}


template<typename Type>
LinAlg::Matrix<3, 4, Type> computeProjectionMatrixFromWorldspacePoints(const std::vector<LinAlg::Vector<3, Type> > &points2D, const std::vector<LinAlg::Vector<4, Type> > &points3D)
{

    LinAlg::Matrix<3,3,Type> T_2D = getCondition<2>(points2D);
    LinAlg::Matrix<4,4,Type> T_3D = getCondition<3>(points3D);

    std::vector<LinAlg::Vector<3,Type> > conditionedPoints2D;
    std::vector<LinAlg::Vector<4,Type> > conditionedPoints3D;

    transform<2>(T_2D, points2D, conditionedPoints2D);
    transform<3>(T_3D, points3D, conditionedPoints3D);

    std::vector<LinAlg::Vector<12, Type> > designMatrix = getDesignMatrix_projectionMatrixFromWSPoints<Type, std::allocator<LinAlg::Vector<12, Type> > >(conditionedPoints2D, conditionedPoints3D);

    LinAlg::Vector<12, Type> singularValues(LinAlg::Vector<12, Type>::NO_INITIALIZATION);
    LinAlg::Matrix<12, 12, Type> singularVectors(LinAlg::Matrix<12, 12, Type>::NO_INITIALIZATION);

    LinAlg::SVD::computeRightSingularVectorsForRealMatrix<std::vector<LinAlg::Vector<12, Type> >, 12, Type>(designMatrix, singularValues, singularVectors);


    unsigned smallestSV = 0;
    for (unsigned i = 1; i < 12; i++) {
        if (singularValues[i] < singularValues[smallestSV])
            smallestSV = i;
    }


    LinAlg::Vector<12, Type> projectionMatrixAsVector = singularVectors.T()[smallestSV];

    LinAlg::Matrix<3, 4, Type> projectionMatrix(LinAlg::Matrix<3, 4, Type>::NO_INITIALIZATION);
    memcpy(&projectionMatrix[0][0], &projectionMatrixAsVector[0], 12*sizeof(Type));


    decondition<Type>(T_2D, T_3D, projectionMatrix);

    return projectionMatrix;
}



template<typename Type>
void refineProjectionMatrixFromWorldspacePoints(const std::vector<LinAlg::Vector<3, Type> > &points2D, const std::vector<LinAlg::Vector<4, Type> > &points3D,
                                LinAlg::Matrix<3, 4, Type> &projectionMatrix)
{
    double lastError = 10000.0;
    double stepsize = 0.1;
    for (unsigned iterations = 0; iterations < 10000; iterations++) {
        LinAlg::Matrix<3, 4, Type> gradient;
        memset(&gradient, 0, 3*4*sizeof(Type));

        double error = 0.0;
        unsigned gradDivisor = 0;
        for (unsigned i = 0; i < points2D.size(); i++) {
            LinAlg::Vector<2, Type> y = points2D[i].StripHom();
            LinAlg::Vector<3, Type> x = points3D[i].StripHom();

            LinAlg::Vector<3, Type> z = projectionMatrix * x.AddHom(1.0);

            if (std::abs(z[2]) < 1e-20)
                continue;

            LinAlg::Vector<2, Type> zNorm = z.StripHom() / z[2];
            LinAlg::Vector<2, Type> d = zNorm - y;

            error += d.SQRLen();

            gradient[0][0] += x[0] * d[0] / z[2];
            gradient[0][1] += x[1] * d[0] / z[2];
            gradient[0][2] += x[2] * d[0] / z[2];
            gradient[0][3] += 1.0  * d[0] / z[2];

            gradient[1][0] += x[0] * d[1] / z[2];
            gradient[1][1] += x[1] * d[1] / z[2];
            gradient[1][2] += x[2] * d[1] / z[2];
            gradient[1][3] += 1.0  * d[1] / z[2];

            gradient[2][0] += x[0] * (-z[0] * d[0] - z[1] * d[1]) / (z[2] * z[2]);
            gradient[2][1] += x[1] * (-z[0] * d[0] - z[1] * d[1]) / (z[2] * z[2]);
            gradient[2][2] += x[2] * (-z[0] * d[0] - z[1] * d[1]) / (z[2] * z[2]);
            gradient[2][3] += 1.0  * (-z[0] * d[0] - z[1] * d[1]) / (z[2] * z[2]);
            gradDivisor++;
        }

        if (gradDivisor == 0)
            return;

        error /= gradDivisor;
        gradient *= 1.0 / gradDivisor;

        if (error <= lastError) {
            stepsize *= 1.2;
        } else {
            stepsize *= 0.5;
        }
        lastError = error;
        /*
        if ((iterations == 0) || (iterations+1 == 100))
            std::cout << error << "  " <<  stepsize << std::endl;
        */
        projectionMatrix -= gradient * stepsize;
    }
}




template<typename Type>
LinAlg::Matrix<3, 4, Type> buildProjectionMatrix(const LinAlg::Matrix<3, 3, Type> &essential, Type scale, unsigned variation)
{
    LinAlg::Vector<3, Type> singularValues;
    LinAlg::Matrix<3, 3, Type> singularVectors;

    LinAlg::SVD::computeRightSingularVectorsForRealMatrixFixedRows<3, 3, Type>(essential, singularValues, singularVectors);


    unsigned smallest = 0;
    for (unsigned i = 1; i < 3; i++)
        if (singularValues[i] < singularValues[smallest])
            smallest = i;

    LinAlg::Matrix<3, 3, Type> leftSVecs = essential * singularVectors *
                    LinAlg::DiagonalMatrix<3, Type>(LinAlg::Fill<Type>((fabs(singularValues[0]) > 1e-20f?1/singularValues[0]:0.0),
                                                                       (fabs(singularValues[1]) > 1e-20f?1/singularValues[1]:0.0),
                                                                       (fabs(singularValues[2]) > 1e-20f?1/singularValues[2]:0.0)));


    LinAlg::Vector3d a, b;
    unsigned indexA, indexB;
    /*
    indexA = smallest==0?1:0;
    indexB = smallest==2?1:2;
    */
    switch (smallest) {
        case 0:
            indexA = 1;
            indexB = 2;
        break;
        case 1:
            indexA = 2;
            indexB = 0;
        break;
        default:
            indexA = 0;
            indexB = 1;
        break;
    }
    for (unsigned i = 0; i < 3; i++) {
        a[i] = leftSVecs[i][indexA];
        b[i] = leftSVecs[i][indexB];
    }
    LinAlg::Vector3d c = LinAlg::cross(a, b);
    for (unsigned i = 0; i < 3; i++) {
        leftSVecs[i][smallest] = c[i];
    }


    LinAlg::Matrix<3, 3, Type> W;
    W[0][0] = 0.0;
    W[1][1] = 0.0;
    W[0][1] = -1.0;
    W[1][0] = 1.0;

    LinAlg::Matrix<3, 3, Type> WInv;
    WInv[0][0] = 0.0;
    WInv[1][1] = 0.0;
    WInv[0][1] = 1.0;
    WInv[1][0] = -1.0;

    LinAlg::Matrix<3, 3, Type> V;
    V[0][0] = 0.0;
    V[1][1] = 0.0;
    V[2][2] = 0.0;
    V[0][1] = -1.0;
    V[1][0] = 1.0;

/*
    std::cout << "U = " << leftSVecs << std::endl;
    std::cout << "S = " << (std::string) singularValues << std::endl;
    std::cout << "V = " << singularVectors << std::endl;
*/

    LinAlg::Matrix<3, 3, Type> TSkew = singularVectors * V * singularVectors.T();
    //LinAlg::Matrix<3, 3, Type> TSkew = singularVectors * W * LinAlg::DiagonalMatrix<3, double>(singularValues) * singularVectors.T();

    //std::cout << TSkew << std::endl;

    LinAlg::Vector<3, Type> T;
    T[0] = -(TSkew[1][2] - TSkew[2][1]) * 0.5;
    T[1] = (TSkew[0][2] - TSkew[2][0]) * 0.5;
    T[2] = -(TSkew[0][1] - TSkew[1][0]) * 0.5;
    //std::cout << (std::string) T << std::endl;

    if (variation&1)
        T *= -1.0;

    T *= scale;
/*
    std::cout << "E = " << essential << std::endl;
    std::cout << "U = " << leftSVecs << std::endl;
    std::cout << "V = " << singularVectors << std::endl;
    std::cout << "WInv = " << WInv << std::endl;
*/

    //Winv.GaussJordanInvert();

    LinAlg::Matrix<3, 3, Type> R;

    if (variation & 2)
        R = leftSVecs * W * singularVectors.T();
    else
        R = leftSVecs * WInv * singularVectors.T();

    T = R * T;
/*
    R[0] *= 1.0f / sqrtf(R[0].SQRLen());
    R[1] *= 1.0f / sqrtf(R[1].SQRLen());
    R[2] *= 1.0f / sqrtf(R[2].SQRLen());
*/
/*
    std::cout << "R" << std::endl;
    std::cout << (std::string) R[0] << std::endl;
    std::cout << (std::string) R[1] << std::endl;
    std::cout << (std::string) R[2] << std::endl;
*/
    float detR = LinAlg::determinant(R);
    if (std::abs(detR) < 1e-20f)
        throw std::runtime_error("Determinant zero!");
    R /= detR;
/*
    std::cout << "R = " << R << std::endl;
    std::cout << "det(R) = " << LinAlg::determinant(R) << std::endl;
*/

    LinAlg::Matrix<3, 4, Type> P;

    for (unsigned i = 0; i < 3; i++)
        P[i] = R[i].AddHom(T[i]);
/*
    std::cout << "P" << std::endl;
    std::cout << (std::string) P[0] << std::endl;
    std::cout << (std::string) P[1] << std::endl;
    std::cout << (std::string) P[2] << std::endl;
*/
    return P * (1.0 / sqrtf(P[2].SQRLen()));
}

template<typename Type>
LinAlg::Matrix<3,3,Type> dropColumn(const LinAlg::Matrix<3, 4, Type> &projectionMatrix, unsigned col)
{
    LinAlg::Matrix<3,3,Type> P;
    for (unsigned i = 0; i < 3; i++) {
        for (unsigned j = 0; j < col; j++)
            P[i][j] = projectionMatrix[i][j];
        for (unsigned j = col+1; j < 4; j++)
            P[i][j-1] = projectionMatrix[i][j];
    }
    return P;
}

template<unsigned dim, typename type>
void GramSchmidtRQDecompose(const LinAlg::Matrix<dim, dim, type> &M, LinAlg::Matrix<dim, dim, type> &R, LinAlg::Matrix<dim, dim, type> &Q)
{
    for (unsigned i = 0; i < dim; i++) {
        LinAlg::Vector<dim, type> v = M[dim-1-i];
        Q[dim-1-i] = v;
        for (unsigned j = 0; j < i; j++) {
            Q[dim-1-i] -= Q[dim-1-j] * (Q[dim-1-j] * v);
        }
        type len = sqrtf(Q[dim-1-i].SQRLen());
        if (len > 1e-20f)
            Q[dim-1-i] *= (type(1) / len);
    }
    R = M * Q.T();
}

template<typename Type>
LinAlg::Vector<3, Type> computeFocalPoint(const LinAlg::Matrix<3, 4, Type> &projectionMatrix)
{
    LinAlg::Vector<3, Type> cameraPosition;

    LinAlg::Matrix<3, 3, Type> M = dropColumn(projectionMatrix, 3);
    Type lambda = std::sqrt(projectionMatrix[2].StripHom().SQRLen());
    if (LinAlg::determinant(M) < 0.0f)
        lambda = -lambda;

    LinAlg::Matrix<3, 4, Type> scaledProjectionMatrix = projectionMatrix;
    scaledProjectionMatrix *= (1.0f /  lambda);
    M *= (1.0f /  lambda);


    Type Cx = LinAlg::determinant(dropColumn(scaledProjectionMatrix, 0));
    Type Cy = -LinAlg::determinant(dropColumn(scaledProjectionMatrix, 1));
    Type Cz = LinAlg::determinant(dropColumn(scaledProjectionMatrix, 2));
    Type Cw = -LinAlg::determinant(dropColumn(scaledProjectionMatrix, 3));

    cameraPosition[0] = Cx / Cw;
    cameraPosition[1] = Cy / Cw;
    cameraPosition[2] = Cz / Cw;

    return cameraPosition;
}


template<typename Type>
void decomposeProjectionMatrix(const LinAlg::Matrix<3, 4, Type> &projectionMatrix,
                                LinAlg::Matrix<3, 3, Type> &internalMatrix,
                                LinAlg::Matrix<4, 4, Type> &worldToEyeSpace,
                                LinAlg::Matrix<4, 4, Type> &cameraRotation,
                                LinAlg::Vector<3, Type> &cameraPosition)
{
    LinAlg::Matrix<3, 3, Type> M = dropColumn(projectionMatrix, 3);
    Type lambda = std::sqrt(projectionMatrix[2].StripHom().SQRLen());
    if (LinAlg::determinant(M) < 0.0f)
        lambda = -lambda;

    LinAlg::Matrix<3, 4, Type> scaledProjectionMatrix = projectionMatrix;
    scaledProjectionMatrix *= (1.0f /  lambda);
    M *= (1.0f /  lambda);


    Type Cx = LinAlg::determinant(dropColumn(scaledProjectionMatrix, 0));
    Type Cy = -LinAlg::determinant(dropColumn(scaledProjectionMatrix, 1));
    Type Cz = LinAlg::determinant(dropColumn(scaledProjectionMatrix, 2));
    Type Cw = -LinAlg::determinant(dropColumn(scaledProjectionMatrix, 3));

    cameraPosition[0] = Cx / Cw;
    cameraPosition[1] = Cy / Cw;
    cameraPosition[2] = Cz / Cw;

    LinAlg::Matrix<3,3,Type> R, Q;
    PCV::GramSchmidtRQDecompose<3, Type>(M, R, Q);


    for (unsigned i = 0; i < 3; i++) {
        if (R[i][i] < Type(0.0)) {
            for (unsigned j = 0; j < 3; j++) {
                R[j][i] = -R[j][i];
                Q[i][j] = -Q[i][j];
            }
        }
    }

    internalMatrix = R;

    LinAlg::Vector<3, Type> revPosition = Q * cameraPosition;
    worldToEyeSpace[0] = Q[0].AddHom(-revPosition[0]);
    worldToEyeSpace[1] = Q[1].AddHom(-revPosition[1]);
    worldToEyeSpace[2] = Q[2].AddHom(-revPosition[2]);

    LinAlg::Matrix<3,3,Type> QT = Q.T();
    cameraRotation[0] = QT[0].AddHom(0.0);
    cameraRotation[1] = QT[1].AddHom(0.0);
    cameraRotation[2] = QT[2].AddHom(0.0);
}


template<typename Type>
LinAlg::Matrix<4, 3, Type> computePseudoInverse(const LinAlg::Matrix<3, 4, Type> &f)
{
    LinAlg::Matrix<4, 4, Type> rightSVecs;
    LinAlg::Vector<4, Type> rightSVals;

    LinAlg::SVD::computeRightSingularVectorsForRealMatrixFixedRows<3, 4, Type>(f, rightSVals, rightSVecs);

    LinAlg::Vector<4, Type> rightSValsInv = LinAlg::Fill<Type>(
            fabs(rightSVals[0])<1e-10?0.0:1.0 / rightSVals[0], fabs(rightSVals[1])<1e-10?0.0:1.0 / rightSVals[1], fabs(rightSVals[2])<1e-10?0.0:1.0 / rightSVals[2], fabs(rightSVals[3])<1e-10?0.0:1.0 / rightSVals[3]);


    LinAlg::Matrix<3, 4, Type> leftSVecs = f * rightSVecs * LinAlg::DiagonalMatrix<4, Type>(rightSValsInv);

    //LinAlg::Matrix<3, 4, Type> test = leftSVecs * LinAlg::DiagonalMatrix<4, Type>(rightSVals) * rightSVecs.T();

    LinAlg::Matrix<4, 3, Type> finv = rightSVecs * LinAlg::DiagonalMatrix<4, Type>(rightSValsInv) * leftSVecs.T();

    return finv;
}


LinAlg::Matrix3x3f computeFundamental(const LinAlg::Matrix3x4f &P1, const LinAlg::Matrix3x4f &P2);



template<typename Type>
LinAlg::Matrix<4, 4, Type> computeHelmertTransformation(const std::vector<LinAlg::Vector<3, Type> > &src, const std::vector<LinAlg::Vector<3, Type> > &dst)
{
    assert(src.size() == dst.size());

    LinAlg::Vector<3, Type> srcCenter;
    LinAlg::Vector<3, Type> dstCenter;

    for (unsigned i = 0; i < src.size(); i++) {
        srcCenter += src[i];
        dstCenter += dst[i];
    }

    srcCenter /= src.size();
    dstCenter /= src.size();


    Type scale;
    {
        Type srcSize = 0.0f;
        Type dstSize = 0.0f;
        for (unsigned i = 0; i < src.size(); i++) {
            srcSize += (src[i] - srcCenter).SQRLen();
            dstSize += (dst[i] - dstCenter).SQRLen();
        }
        scale = std::sqrt(dstSize / srcSize);
    }


    LinAlg::Matrix<3, 3, Type> M(LinAlg::Matrix<3, 3, Type>::NO_INITIALIZATION);
    for (unsigned i = 0; i < 3; i++)
        for (unsigned j = 0; j < 3; j++)
            M[i][j] = Type(0);

    for (unsigned i = 0; i < src.size(); i++) {
        LinAlg::Vector<3, Type> s = src[i] - srcCenter;
        LinAlg::Vector<3, Type> d = dst[i] - dstCenter;

        for (unsigned k = 0; k < 3; k++)
            for (unsigned l = 0; l < 3; l++)
                M[k][l] += s[k] * d[l];

    }


    LinAlg::Matrix<4, 4, Type> N(LinAlg::Matrix<4, 4, Type>::NO_INITIALIZATION);

    N[0][0] = M[0][0] + M[1][1] + M[2][2];
    N[0][1] = M[1][2] - M[2][1];
    N[0][2] = M[2][0] - M[0][2];
    N[0][3] = M[0][1] - M[1][0];

    N[1][0] = M[1][2] - M[2][1];
    N[1][1] = M[0][0] - M[1][1] - M[2][2];
    N[1][2] = M[0][1] + M[1][0];
    N[1][3] = M[2][0] + M[0][2];

    N[2][0] = M[2][0] - M[0][2];
    N[2][1] = M[0][1] + M[1][0];
    N[2][2] = -M[0][0] + M[1][1] - M[2][2];
    N[2][3] = M[1][2] + M[2][1];

    N[3][0] = M[0][1] - M[1][0];
    N[3][1] = M[2][0] + M[0][2];
    N[3][2] = M[1][2] + M[2][1];
    N[3][3] = -M[0][0] - M[1][1] + M[2][2];


    LinAlg::Matrix<4, 4, Type> eigenVectors;
    LinAlg::Vector<4, Type> eigenValues;

    LinAlg::computeEigenValues<4, Type>(N, eigenValues, &eigenVectors);

    unsigned biggest = 0;
    for (unsigned i = 1; i < 4; i++)
        if (eigenValues[i] > eigenValues[biggest])
            biggest = i;

    Type real = eigenVectors[0][biggest];
    Type imag1 = eigenVectors[1][biggest];
    Type imag2 = eigenVectors[2][biggest];
    Type imag3 = eigenVectors[3][biggest];


    LinAlg::Matrix<4, 4, Type> rotation;

    rotation[0][0] = real*real + imag1*imag1 - imag2*imag2 - imag3*imag3;
    rotation[0][1] = Type(2) * (imag1*imag2 - real*imag3);
    rotation[0][2] = Type(2) * (imag1*imag3 + real*imag2);

    rotation[1][0] = Type(2) * (imag2*imag1 + real*imag3);
    rotation[1][1] = real*real - imag1*imag1 + imag2*imag2 - imag3*imag3;
    rotation[1][2] = Type(2) * (imag2*imag3 - real*imag1);

    rotation[2][0] = Type(2) * (imag3*imag1 - real*imag2);
    rotation[2][1] = Type(2) * (imag3*imag2 + real*imag1);
    rotation[2][2] = real*real - imag1*imag1 - imag2*imag2 + imag3*imag3;

    return LinAlg::Translation3D(dstCenter) * LinAlg::Scale3D(LinAlg::Fill(scale, scale, scale)) * rotation * LinAlg::Translation3D(srcCenter.negated());
}




}


#endif // PCVTOOLBOX_HPP_INCLUDED
