/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "BundlerExporter.h"
#include <SFM/SFM.h>
#include <SFM/Frame.h>
#include <SFM/InternalCameraCalibration.h>
#include <SFM/Track.h>
#include <SFM/TrackObservation.h>


#include <fstream>

namespace SFM {
namespace Utilities {

void exportBundler(const boost::filesystem::path &destinationPath, SFM &sfm)
{
    unsigned numTracks = 0;
    for (const Track &track : sfm.getTrackList()) {
        if (track.getState() == Track::STATE_DISABLED) continue;
        numTracks++;
    }

    std::fstream bundlerFile;
    bundlerFile.exceptions(std::ifstream::failbit | std::ifstream::badbit);
    bundlerFile.open(destinationPath.native().c_str(), std::fstream::out);
    bundlerFile << "# Bundle file v0.3" << std::endl;
    bundlerFile << sfm.getFrames().size() << " " << numTracks << std::endl;
    for (unsigned i = 0; i < sfm.getFrames().size(); i++) {

        Frame &frame = *sfm.getFrames()[i];

        const float approxFocalLength = (frame.getCamera().getInternalCalibration()->getProjectionMatrix()[0][0] +
                                         frame.getCamera().getInternalCalibration()->getProjectionMatrix()[1][1]) * 0.5f * 0.5f * frame.getWidth();

        bundlerFile << approxFocalLength << " 0 0" << std::endl;


        LinAlg::Matrix4x4f viewMatrix = LinAlg::Scale3D(LinAlg::Fill(-1.0f, 1.0f, 1.0f)) * frame.getCamera().getViewMatrix();

        bundlerFile << viewMatrix[0][0] << " " << viewMatrix[0][1] << " " << viewMatrix[0][2] << std::endl;
        bundlerFile << viewMatrix[1][0] << " " << viewMatrix[1][1] << " " << viewMatrix[1][2] << std::endl;
        bundlerFile << viewMatrix[2][0] << " " << viewMatrix[2][1] << " " << viewMatrix[2][2] << std::endl;

        bundlerFile << viewMatrix[0][3] << " " << viewMatrix[1][3] << " " << viewMatrix[2][3] << std::endl;

    }
    for (const Track &track : sfm.getTrackList()) {
        if (track.getState() == Track::STATE_DISABLED) continue;

        LinAlg::Vector3f euclPosition = track.getLastWSPositionEstimate().StripHom() / track.getLastWSPositionEstimate()[3];

        bundlerFile << euclPosition[0] << " " << euclPosition[1] << " " << euclPosition[2] << std::endl;
        bundlerFile << 255 << " " << 255 << " " << 255 << std::endl;

        std::vector<const TrackObservation*> obs;
        for (const TrackObservation &observation : track.getObservations()) {
            if (observation.isFaulty()) continue;
            obs.push_back(&observation);
        }
        bundlerFile << obs.size();
        for (unsigned j = 0; j < obs.size(); j++) {
            bundlerFile << " " << obs[j]->getFrame()->getIndex() << " " << obs[j]->getFrameFeaturePointIndex()
                        << " " << -obs[j]->getUndistortedScreenSpacePosition()[0] * obs[j]->getFrame()->getWidth() * 0.5f
                        << " " << obs[j]->getUndistortedScreenSpacePosition()[1] * obs[j]->getFrame()->getHeight() * 0.5f;

        }
        bundlerFile << std::endl;
    }
}

}
}
