/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "ExtractBasicColorsForTracks.h"

#include <SFM/SFM.h>
#include <SFM/Track.h>
#include <SFM/TrackObservation.h>
#include <SFM/Frame.h>

namespace SFM {
class SFM;
namespace Utilities {

void extractBasicColorsForTracks(const SFM &sfm,
                                 std::vector<LinAlg::Vector3f> &colors,
                                 const std::function<bool(float)> &progressCallback)
{

    colors.resize(sfm.getTrackList().reservedSize());
    std::vector<float> sums;
    sums.resize(sfm.getTrackList().reservedSize());

    std::vector<const TrackObservation*> observations;
    for (unsigned frameIdx = 0; frameIdx < sfm.getFrames().size(); frameIdx++) {
        observations.clear();

        for (const Track &track : sfm.getTrackList()) {
            if (track.getState() == Track::STATE_DISABLED)
                continue;

            for (const TrackObservation &obs : track.getObservations()) {
                if (obs.isFaulty())
                    continue;

                if (obs.getFrame() == sfm.getFrames()[frameIdx].get()) {
                    observations.push_back(&obs);
                }
            }
        }

        RasterImage image;
        image.loadFromFile(sfm.getFrames()[frameIdx]->getImageFilename().c_str());

        for (unsigned i = 0; i < observations.size(); i++) {
            const Track &track = *observations[i]->getTrack();

            const LinAlg::Vector2f &screenPos = observations[i]->getUndistortedScreenSpacePosition();
            int screenX = std::min(std::max<int>(screenPos[0] * 0.5f * image.getWidth() + 0.5f * image.getWidth(), 0), (int)image.getWidth()-1);
            int screenY = std::min(std::max<int>(screenPos[1] * 0.5f * image.getWidth() + 0.5f * image.getHeight(), 0), (int)image.getHeight()-1);

            uint32_t color = image.getData()[screenX + screenY * image.getWidth()];

            LinAlg::Vector3f vecColor;
            vecColor[0] = (color >> 0) & 0xFF;
            vecColor[1] = (color >> 8) & 0xFF;
            vecColor[2] = (color >> 16) & 0xFF;

            int trackIndex = sfm.getTrackList().findIndexOf(&track);
            assert(trackIndex != -1);
            colors[trackIndex] += vecColor;
            sums[trackIndex] += 255.0f;
        }
    }

    for (unsigned i = 0; i < sums.size(); i++) {
        if (sums[i] > 0.0f)
            colors[i] *= 1.0f / sums[i];
    }
}

}
}
