/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef IMAGERESAMPLER_H
#define IMAGERESAMPLER_H

#include <vector>
#include <string>
#include <tools/LinAlg.h>
#include <SFM/BundleAdjustment/BundleAdjustment.h>
#include <config/CudaConfig.h>

#include <cudaUtilities/CudaCodeModule.h>

namespace CudaUtils {
    class CudaTextureReference;
    class CudaSurfaceReference;
    class CudaKernel;
}

namespace SFM {
namespace Utilities {

/**
 * @brief Implements the resampling of batches of images.
 * @details Can be used to undistort images with the distortion parameter
 * optained from the SfM pipeline.
 * @ingroup SFMUtilities_Group
 */
class ImageResampler
{
    public:
        ImageResampler(const config::CudaConfig &cudaConfig);

        struct ProgressReport {
            unsigned currentImage;
            unsigned totalNumImages;
        };
        typedef std::function<bool(const ProgressReport&)> ProgressReportCallback;

        void undistortImages(const std::vector<std::pair<std::string, std::string> > &filenames,
                             const BundleAdjustment::RadialDistortionParametrization &distortion,
                             const ProgressReportCallback &progressReportCallback = ProgressReportCallback());

        void resampleHomography(const std::vector<std::pair<std::string, std::string> > &filenames,
                             const LinAlg::Matrix4x4f &homography,
                             const ProgressReportCallback &progressReportCallback = ProgressReportCallback());

    private:
        CudaUtils::CudaCodeModule m_codeModule;
        std::unique_ptr<CudaUtils::CudaTextureReference> m_inputTexRef;
        std::unique_ptr<CudaUtils::CudaSurfaceReference> m_outputSurfRef;

        std::unique_ptr<CudaUtils::CudaKernel> m_kernelRadialPolynomial234;
        std::unique_ptr<CudaUtils::CudaKernel> m_kernelHomography;

        void copyImages(const std::vector<std::pair<std::string, std::string> > &filenames, const ProgressReportCallback &progressReportCallback);
        void undistortImagesRadialPolynomial234(const std::vector<std::pair<std::string, std::string> > &filenames, const float *kappa, const ProgressReportCallback &progressReportCallback);


        static void copyFile(const char *dst, const char *src);
};

}
}

#endif // IMAGERESAMPLER_H
