/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "PMVSExporter.h"

#include <config/CudaConfig.h>

#include <SFM/SFM.h>
#include <SFM/Frame.h>
#include <SFM/InternalCameraCalibration.h>

#include "ImageResampler.h"

#include <sstream>
#include <iomanip>
#include <fstream>

namespace SFM {
namespace Utilities {

void exportPMVS(const boost::filesystem::path &destinationPath, SFM &sfm)
{
    config::CudaConfig cudaConfig;
    ImageResampler imageResampler(cudaConfig);

    boost::filesystem::path modelPath = destinationPath/"models";
    boost::filesystem::path txtPath = destinationPath/"txt";
    boost::filesystem::path visualizePath = destinationPath/"visualize";

    boost::filesystem::create_directories(modelPath);
    boost::filesystem::create_directories(txtPath);
    boost::filesystem::create_directories(visualizePath);

    BundleAdjustment::RadialDistortionParametrization distortion;

    std::vector<std::pair<std::string, std::string> > filenames;
    filenames.reserve(sfm.getFrames().size());
    for (unsigned i = 0; i < sfm.getFrames().size(); i++) {
        Frame &frame = *sfm.getFrames()[i];
        if (!frame.active())
            continue;

        distortion = frame.getCamera().getInternalCalibration()->getRadialDistortion();

        std::pair<std::string, std::string> srcDstFilenamePair;
        srcDstFilenamePair.first = frame.getImageFilename();

        std::stringstream number;
        number << std::setw(8) << std::setfill('0') << filenames.size();

        srcDstFilenamePair.second = (visualizePath / (number.str() + ".jpg")).native();


        {
            std::fstream cameraFile;
            cameraFile.exceptions(std::ifstream::failbit | std::ifstream::badbit);
            cameraFile.open((txtPath / (number.str() + ".txt")).native().c_str(), std::fstream::out);

            cameraFile << "CONTOUR" << std::endl;

            LinAlg::Matrix3x4f PVmatrix = frame.getCamera().getProjectionViewMatrix().dropRow(2);

            LinAlg::Matrix3x3f imageMatrix;
            imageMatrix[0][0] = frame.getWidth() * 0.5f;
            imageMatrix[1][1] = frame.getWidth() * 0.5f;

            imageMatrix[0][2] = frame.getWidth() * 0.5f - 0.5f;
            imageMatrix[1][2] = frame.getHeight() * 0.5f - 0.5f;

            PVmatrix = imageMatrix * PVmatrix;

            cameraFile << std::setprecision(15) << PVmatrix[0][0] << " " <<
                                                   PVmatrix[0][1] << " " <<
                                                   PVmatrix[0][2] << " " <<
                                                   PVmatrix[0][3] << std::endl;

            cameraFile << std::setprecision(15) << PVmatrix[1][0] << " " <<
                                                   PVmatrix[1][1] << " " <<
                                                   PVmatrix[1][2] << " " <<
                                                   PVmatrix[1][3] << std::endl;

            cameraFile << std::setprecision(15) << PVmatrix[2][0] << " " <<
                                                   PVmatrix[2][1] << " " <<
                                                   PVmatrix[2][2] << " " <<
                                                   PVmatrix[2][3] << std::endl;
            cameraFile.close();
        }


        filenames.push_back(srcDstFilenamePair);
    }

    std::fstream optionsFile;
    optionsFile.exceptions(std::ifstream::failbit | std::ifstream::badbit);
    optionsFile.open((destinationPath/"options.txt").native().c_str(), std::fstream::out);
    optionsFile << "level 1" << std::endl
                << "csize 2" << std::endl
                << "threshold 0.7" << std::endl
                << "wsize 7" << std::endl
                << "minImageNum 3" << std::endl
                << "CPU 8" << std::endl
                << "setEdge 0" << std::endl
                << "useBound 0" << std::endl
                << "useVisData 0" << std::endl
                << "sequence -1" << std::endl
                << "maxAngle 10" << std::endl
                << "quad 2.0" << std::endl
                << "timages -1 0 " << filenames.size() << std::endl
                << "oimages 0" << std::endl;


    imageResampler.undistortImages(filenames, distortion);
}

}
}
